;+
;                  *** The ADIENT Data Collocator ***
; pro adient_data_collocator, faam_core, modisl2=modisl2, aatsr=aatsr, $
;                            meris=meris, seviri=seviri, misr=misr,    $
;                            maxdist=maxdist, maxtime=maxtime,         $
;                            ftime=ftime, track=track, save=save,      $
;                            llimit=llimit, tlimit=tlimit,             $
;                            modisdir=modisdir, aatsrdir=aatsrdir,     $
;                            merisdir=merisdir, seviridir=seviridir,   $
;                            misrdir=misrdir
;
; Reads location and time information from FAAM core (v002) data and
; extracts co-located satellite data for a range of sensors. Data is 
; returned as a structure for each instrument, or it can be output into
; single NetCDF file.
;
; INPUT ARGUMENT
; faam_core  The filename (with path) to the FAAM core data from which
;            to extract the aircraft path.
; KEYWORDS
; modisl2=modisl2 If specified MODIS L2 data will be read. On return
;            the named variable will contain the extracted MODIS data.
; aatsr=aatsr If specified AATSR (GlobAEROSOL format) data will be read.
;            On return the  named variable will contain the extracted
;            data.
; meris=meris If specified MERIS L2 data will be read. On return the
;            named variable will contain the extracted data.
; seviri=seviri If specified AATSR (GlobAEROSOL format) data will be
;            read. On return the  named variable will contain the
;            extracted data.
; misr=misr  If specified MISR (L2?) data will be read. On return the
;            named variable will contain the extracted data.
; maxdist=maxdist Specifies the maximum distance over which to accept
;            satellite data as collocated with the aircraft
;            measurement. All data in this range will be returned for
;            each sample along the flight path. Defaults to +/-10 km.
; maxtime=maxtime Specifies the maximum time window within which
;            satellite data will be counted as collocated with the 
;            aircraft. All data in this range will be returned for
;            each sample along the flight path. Defaults to +/-60 mins.
; ftime=ftime Temporal spacing of samples to take along the flight path.
;            Defaults to 30 mins. Should be specified in minutes.
; track=track On return will contain the actual points sampled from the
;            flight path (lat, lon, time)
; save=save  If specified the extracted data will be saved into a
;            NetCDF file for later analysis.
; llimit=llimit Allows a limited geographical region to be analysed,
;            rather than the entire flight. Should be an array of this
;            format: [min_lat, min_lon, max_lat, max_lon]
; tlimit=tlimit Allows a limited time period to be analysed, rather
;            than the entire flight. Should be a two element array:
;            [min_sec, max_sec] in units of seconds since the start of
;            the day (UT).
; modisdir=modisdir
; aatsrdir=aatsrdir
; merisdir=merisdir
; seviridir=seviridir
; misrdir=misrdir Allow the directory for data from each instrument to
;            be specified.
; HISTORY
; 02/12/08 G Thomas: Original, first attempt with support for MODIS L2
;            only.
;-

function calc_dayofyear,yr,mt,dy
; Calculates the day of year, as used by MODIS etc, from a calendar
; date
; 1st of January is day 1.
  dl = julday(mt,dy,yr) - julday(1,1,yr)+1
  if dl ge 100 then d = strtrim(dl,2) $
  else if dl ge 10 then d = '0'+strtrim(dl,2) $
  else d = '00'+strtrim(dl,2)
  return, d
end

function concat_struct,s1,s2,dim=dim,exceptions=exceptions
; Concatinates two structures (with the same format) into a single
; one. If a structure contains arrays with more than one dimension,
; the dim keyword can be used to define which dimension will be
; extended by the concatination.
  tags1 = tag_names(s1)
  tags2 = tag_names(s2)
  if ~array_equal(tags1,tags2) then $
     message,'CONCAT_STRUCT: Structures do not have same contents'

  if n_elements(dim) eq 0 then dim = 0

  ntags = n_elements(tags1)
; Check if any of the variables within the structures are themselves
; structures. Combine them first if they are. Note, only one layer of
; substructures is supported at the moment.
  for i=0,ntags-1 do begin
     tsz1 = size(s1.(i))
     tsz2 = size(s2.(i))
     if tsz1[tsz1[0]+1] eq 8 then begin
        tags3 = tag_names(s1.(i))
        for j=0,n_elements(tags3)-1 do begin
           asz1 = size(s1.(i).(j))
           asz2 = size(s2.(i).(j))
           isexcept = where(exceptions.tags eq tags1[i]+'.'+tags3[j])
           isexcept = isexcept[0]
           if asz[0] ge dim+1 and isexcept lt 0 then begin
;             This array is to concatinated using the dim argument
              tmpasz = asz1
              tmpasz[dim+1] = asz1[dim+1]+asz2[dim+1]
              tmpa = make_array(size=tmpasz)
              tmpa[0:n_elements(s1.(i).(j))-1] = s1.(i).(j)
              tmpa[n_elements(s1.(i).(j)):*]   = s2.(i).(j)
           endif else if isexcept ge 0 then begin
;             This array is to be concatinated using the dimension given
;             in the exceptions keyword.
              tmpasz = asz1
              tmpasz[exceptions.dims[isexcept]+1] =                    $
                 asz1[exceptions.dim[isexcept]+1]+                     $
                 asz2[exceptions.dim[isexcept]+1]
              tmpa = make_array(size=tmpasz)
              tmpa[0:n_elements(s1.(i).(j))-1] = s1.(i).(j)
              tmpa[n_elements(s1.(i).(j)):*]   = s2.(i).(j)
           endif else begin
;             This array is smaller then dim, and isn't listed in
;             exceptions, so we just concatinate in the 1st dimension
              tmpa = [s1.(i).(j),s2.(i).(j)]
           endelse
           if j eq 0 then tmps = create_struct(tags3[j],tmpa) $
                     else tmps = create_struct(tmps,tags3[j],tmpa)
        endfor
     endif else begin
;       Structure element isn't a substructure, so we can simply
;       concatinate it...
        isexcept = where(exceptions.tags eq tags1[i])
        isexcept = isexcept[0]
        if tsz1[0] ge dim+1 and isexcept lt 0 then begin
;          This array is to concatinated using the dim argument
           tmpssz = tsz1
           tmpssz[dim+1] = tsz1[dim+1]+tsz2[dim+1]
           tmps = make_array(size=tmpssz)
           tmps[0:n_elements(s1.(i))-1] = s1.(i)
           tmps[n_elements(s1.(i)):*]   = s2.(i)
        endif else if isexcept ge 0 then begin
;          This array is to be concatinated using the dimension given in
;          the exceptions keyword.
           tmpssz = tsz1
           tmpssz[exceptions.dims[isexcept]+1] =                       $
              tsz1[exceptions.dims[isexcept]+1] +                      $
              tsz2[exceptions.dims[isexcept]+1]
           tmps = make_array(size=tmpssz)
           tmps[0:n_elements(s1.(i))-1] = s1.(i)
           tmps[n_elements(s1.(i)):*]   = s2.(i)
        endif else begin
;          This array is smaller then dim, and isn't listed in
;          exceptions, so we just concatinate in the 1st dimension
           tmps = [s1.(i),s2.(i)]
        endelse
     endelse
     if i eq 0 then out = create_struct(tags1[i],tmps) $
               else out = create_struct(out,tags1[i],tmps)
  endfor
  return, out
end

function subset_struct, in, idx, dimen
; Applies a subseting (i.e. an array of index numbers) to a structure
; of data, such as generally returned when reading satellite data.
; The argument IN is the structure to be subsetted
; The argument IDX is the array of index numbers to be extracted
; The argument DIMEN is the dimensions of the arrays to apply the
; subsetting to. Only arrays with matching dimensions will be
; subsetted, others will be returned unchanged
  tags = tag_names(in)
  ntags = n_elements(tags)
  ndimen = n_elements(dimen)
  for i=0,ntags-1 do begin
     insz = size(in.(i))
     if insz[0] eq ndimen then begin
        if array_equal(insz[1:insz[0]],dimen) then begin
;          Array has the same dimensions, simply apply the subsetting
           if (size(out))[0] eq 0                                     $
           then out = create_struct(tags[i],(in.(i))[idx])            $
           else out = create_struct(out,tags[i],(in.(i))[idx])
        endif else begin
;          Array has the same number of dimensions, but they don't
;          match what we're looking for. Copy the array into the output
;          unchanged
           if (size(out))[0] eq 0                                     $
           then out = create_struct(tags[i],(in.(i)))                 $
           else out = create_struct(out,tags[i],(in.(i)))
        endelse
     endif else begin
;       Array doesn't have the same number of dimensions. Copy the 
;       array into the output unchanged
        if (size(out))[0] eq 0                                        $
           then out = create_struct(tags[i],(in.(i)))                 $
           else out = create_struct(out,tags[i],(in.(i)))
     endelse
  endfor
  return,out
end

function reform_struct, instr, var
  xdim = n_tags(instr)
  tags = tag_names(instr.(0))
  thisvar = where(tags eq strupcase(var))
  maxi = 0
  if xdim gt 1 then for i=1,xdim-1 do $
     if n_elements(instr.(i).(thisvar)) gt $
     n_elements(instr.(maxi).(thisvar)) then maxi=i
  varsize = size(instr.(maxi).(thisvar))
  outsize = [varsize[0]+1, xdim, varsize[1:varsize[0]+1],              $
             xdim*varsize[varsize[0]+2]]
  outarr = make_array(size=outsize,value=-999.0)
  if varsize[0] eq 1 then begin
     for i=0,xdim-1 do begin
        sizei = n_elements((instr.(i).(thisvar)))
        outarr[i,0:sizei-1] = instr.(i).(thisvar) 
     endfor
  endif else if varsize[0] eq 2 then begin
     for i=0,xdim-1 do begin
        sizei = size((instr.(i).(thisvar)))
        if sizei[0] eq 1 then begin
           outarr[i,0:sizei[1]-1,0] = instr.(i).(thisvar) 
        endif else begin
           outarr[i,0:sizei[1]-1,0:sizei[2]-1] = instr.(i).(thisvar)
        endelse
     endfor
  endif else begin
     for i=0,xdim-1 do begin
        sizei = size((instr.(i).(thisvar)))
        if sizei[0] eq 2 then begin
           outarr[i,0:sizei[1]-1,0:sizei[2]-1,0] = instr.(i).(thisvar)
        endif else begin
           outarr[i,0:sizei[1]-1,0:sizei[2]-1,0:sizei[3]-1] = $
           instr.(i).(thisvar)
        endelse
     endfor
  endelse
  return,outarr
end


pro save_output_data, faam, faamidx, input_files, inst_list,           $
                      modisl2=modisl2,     $
                      aatsr=aatsr, meris=meris, seviri=seviri,         $
                      misr=misr, outfile=outfile
; Outputs the data from the collocator into a NetCDF file
  
; If the file name hasn't been specified, use a default one
; Open the NCDF file and put it in definition mode
  chunks = strsplit(faam.title,/extract)
  flightno = chunks[2]
  flightdate = chunks[4]
  if ~keyword_set(outfile) then $
     outfile = 'FAAM_satellite_collocated_data_'+flightno+'.nc'
  fid = ncdf_create(outfile,/clobber)
  ncdf_control,0,/verbose
; Define some global attributes
  ncdf_attput,fid,/global,'productID',outfile
  ncdf_attput,fid,/global,'title',                                     $
              'Satellite data collocated with FAAM flight '+flightno+   $
              ' on '+flightdate
  ncdf_attput,fid,/global,'institution','AOPP, University of Oxford'
  ncdf_attput,fid,/global,'source','adient_data_collocator'
  ncdf_attput,fid,/global,'references',                                $
              'http://www.atm.ox.ac.uk/project/adient, gthomas@atm.ox.ac.uk'
  ncdf_attput,fid,/global,'sensors',strjoin(inst_list,', ')
  ncdf_attput,fid,/global,'Input_Files',strjoin(input_files,', ')
  ncdf_attput,fid,/global,'Data_Date',faam.data_date
  ncdf_attput,fid,/global,'TimeInterval',faam.timeinterval
; Define the NCDF dimensions
  xid = ncdf_dimdef(fid,'FAAM_samples',n_elements(faamidx))
;  ncdf_attput,fid,xid,'long_name','Samples from FAAM flight path'
  yid = ncdf_dimdef(fid,'Satellite_matches',/unlimited)
;  ncdf_attput,fid,yid,'long_name','Collocated satellite measurements'
  z1id = ncdf_dimdef(fid,'MODIS_ocean_idx',2)
  z2id = ncdf_dimdef(fid,'MODIS_quality_idx',5)
; Define MODIS data arrays, if required
  modisl2exists = 0
  if keyword_set(modisl2) then begin
     modsize = size(modisl2)
     if modsize[n_elements(modsize)-2] eq 8 then modisl2exists = 1
  endif
  if modisl2exists then begin
     modl2latid = ncdf_vardef(fid,'MODL2_Latitude',[xid,yid],/float)
     modl2lonid = ncdf_vardef(fid,'MODL2_Longitude',[xid,yid],/float)
     modl2timid = ncdf_vardef(fid,'MODL2_Scan_Start_Time',[xid,yid],   $
                              /double)
     modl2cmqaid = ncdf_vardef(fid,'MODL2_Cloud_Mask_QA',[xid,yid],    $
                              /float)
     modl2odloid = ncdf_vardef(fid,'MODL2_Optical_Depth_Land_And_Ocean',$
                               [xid,yid],/float)
     modl2iodloid = ncdf_vardef(fid,                                   $
                                'MODL2_Image_Optical_Depth_Land_And_Ocean',$
                                [xid,yid],/float)
     modl2odrid = ncdf_vardef(fid,                                    $
                               'MODL2_Optical_Depth_Ratio_Small_Land_And_Ocean',$
                              [xid,yid],/float)
     modl2atlid = ncdf_vardef(fid,'MODL2_Aerosol_Type_Land',[xid,yid], $
                              /float)
     modl2felid = ncdf_vardef(fid,'MODL2_Fitting_Error_Land',[xid,yid],$
                              /float)
     modl2aelid = ncdf_vardef(fid,'MODL2_Angstrom_Exponent_Land',      $
                              [xid,yid],/float)
     modl2mclid = ncdf_vardef(fid,'MODL2_Mass_Concentration_Land',     $
                              [xid,yid],/float)
     modl2cflid = ncdf_vardef(fid,'MODL2_Cloud_Fraction_Land',         $
                              [xid,yid],/float)
     modl2qalid = ncdf_vardef(fid,'MODL2_Quality_Assurance_Land',      $
                              [xid,z2id,yid],/float)
     modl2mcoid = ncdf_vardef(fid,'MODL2_Mass_Concentration_Ocean',    $
                              [xid,z1id,yid],/float)
     modl2eroid = ncdf_vardef(fid,'MODL2_Effective_Radius_Ocean',      $
                              [xid,z1id,yid],/float)
     modl2ae1oid = ncdf_vardef(fid,'MODL2_Angstrom_Exponent_1_Ocean',  $
                               [xid,z1id,yid],/float)
     modl2cfoid = ncdf_vardef(fid,'MODL2_Cloud_Fraction_Ocean',        $
                              [xid,yid],/float)
     modl2qaoid = ncdf_vardef(fid,'MODL2_Quality_Assurance_Ocean',     $
                              [xid,z2id,yid],/float)
;    Put the NCDF file in datamode and populate the data arrays
     ncdf_control,fid,/endef
     out = reform_struct(modisl2,'latitude')
     ncdf_varput,fid,modl2latid,out
     out = reform_struct(modisl2,'longitude')
     ncdf_varput,fid,modl2lonid,out
     out = reform_struct(modisl2,'Scan_Start_Time')
     ncdf_varput,fid,modl2timid,out
     out = reform_struct(modisl2,'Cloud_Mask_QA')
     ncdf_varput,fid,modl2cmqaid,out
     out = reform_struct(modisl2,'Optical_Depth_Land_And_Ocean')
     ncdf_varput,fid,modl2odloid,out
     out = reform_struct(modisl2,'Image_Optical_Depth_Land_And_Ocean')
     ncdf_varput,fid,modl2iodloid,out
     out = reform_struct(modisl2,'Optical_Depth_Ratio_Small_Land_And_Ocean')
     ncdf_varput,fid,modl2odrid,out
     out = reform_struct(modisl2,'Aerosol_Type_Land')
     ncdf_varput,fid,modl2atlid,out
     out = reform_struct(modisl2,'Fitting_Error_Land')
     ncdf_varput,fid,modl2felid,out
     out = reform_struct(modisl2,'Angstrom_Exponent_Land')
     ncdf_varput,fid,modl2aelid,out
     out = reform_struct(modisl2,'Mass_Concentration_Land')
     ncdf_varput,fid,modl2mclid,out
     out = reform_struct(modisl2,'Cloud_Fraction_Land')
     ncdf_varput,fid,modl2cflid,out
     out = reform_struct(modisl2,'Quality_Assurance_Land')
     ncdf_varput,fid,modl2qalid,out
     out = reform_struct(modisl2,'Mass_Concentration_Ocean')
     ncdf_varput,fid,modl2mcoid,out
     out = reform_struct(modisl2,'Effective_Radius_Ocean')
     ncdf_varput,fid,modl2eroid,out
     out = reform_struct(modisl2,'Angstrom_Exponent_1_Ocean')
     ncdf_varput,fid,modl2ae1oid,out
     out = reform_struct(modisl2,'Cloud_Fraction_Ocean')
     ncdf_varput,fid,modl2cfoid,out
     out = reform_struct(modisl2,'Quality_Assurance_Ocean')
     ncdf_varput,fid,modl2qaoid,out
  endif
; close the NCDF file
  ncdf_close,fid
end


pro adient_data_collocator,faam_core, modisl2=modisl2, aatsr=aatsr,    $
                           meris=meris, seviri=seviri, misr=misr,      $
                           maxdist=maxdist, maxtime=maxtime,           $
                           ftime=ftime, track=track, save=save,        $
                           llimit=llimit, tlimit=tlimit,               $
                           modisdir=modisdir, aatsrdir=aatsrdir,       $
                           merisdir=merisdir, seviridir=seviridir,     $
                           misrdir=misrdir

; Setup default values if keywords aren't specified
  if ~keyword_set(maxdist) then maxdist = 10.0
  if ~keyword_set(maxtime) then maxtime = 60.0
  if ~keyword_set(ftime)   then ftime = 30.0

;               ***** FAAM data extraction section ****
; Load the flight track data
  faamvars = ['LAT_GPS','LON_GPS','Time']
  faam = load_faam_core(faam_core, variables=faamvars, /varattr,       $
                        /globattr)

; Keep track of all files that are read in
  input_files=[faam_core]

; Extract the date of the measurement
  split = strsplit(faam.time.units,' ')
  yr  = fix(strmid(faam.time.units,split[2],4))
  mt  = fix(strmid(faam.time.units,split[2]+5,2))
  dy  = fix(strmid(faam.time.units,split[2]+8,2))
  hr  = fix(strmid(faam.time.units,split[3],2))
  mn  = fix(strmid(faam.time.units,split[3]+3,2))
  sc  = fix(strmid(faam.time.units,split[3]+6,2))
  scoff = double(strmid(faam.time.units,split[4]))
; Calculate Julian date from which timing is taken
  faamjd = julday(mt,dy,yr,hr,mn,sc) + scoff / 86400.0

; Convert FAAM time stamps into Julian dates
  faamtim = faamjd + double(faam.time.data) / 86400.0

; Detect any fill values
  ok = faam.lat_gps.data ne faam.lat_gps._fillvalue and                $
       faam.lon_gps.data ne faam.lon_gps._fillvalue and                $
       faam.time.data ne faam.time._fillvalue

  faamlat = faam.lat_gps.data
  faamlon = faam.lon_gps.data
; Apply location and time limits if required
  if keyword_set(llimit) then begin
     ok = ok and                                                       $
          faamlat ge llimit[0] and faamlat le llimit[2] and            $
          faamlon ge llimit[1] and faamlon le llimit[3]
  endif
  if keyword_set(tlimit) then begin
     ftlim = tlimit * 3600.0
     ok = ok and $
          faam.time.data ge ftlim[0] and faam.time.data le ftlim[1]
  endif

; Check if any data passes the limits and then extract it from the
; arrays
  ok = where(ok, nok)
  if nok eq 0 then begin
     print,'ADIENT_DATA_COLLOCATOR: No valid FAAM data found in specified range.'
     print,'                       Quiting'
     return
  endif else begin
     faamlat = faamlat[ok]
     faamlon = faamlon[ok]
     faamtim = faamtim[ok]
  endelse

  tstep = ftime / 1440.

  tidx = 0
  tim  = min(faamtim)
  while tim lt max(faamtim) do begin
     tim = tim + tstep
     tmp = where(abs(faamtim - tim) eq min(abs(faamtim - tim)))
     tidx = [tidx,tmp]
  endwhile
  faamidx = ok[tidx]
  faamlat = faamlat[tidx]
  faamlon = faamlon[tidx]
  faamtim = faamtim[tidx]
  nfaam = n_elements(faamlat)
; We now have a list of latitudes, longitudes and times to find within
; the satellite data

;            ***** Satellite data extraction section ****
;                              * MODIS MOD04 *
  if arg_present(modisl2) then begin
;    Check if the MODIS_DIR keyword is specified. If it hasn't been,
;    then get it from the MODIS_DIR environment variable. Warn if this
;    is null.
     if ~keyword_set(modisdir) then begin
        modisdir = getenv('MODIS_DIR')
        if strlen(modis_dir) eq 0 then begin
           modisdir='.'
           message,/info,                                              $
                   'ADIENT_DATA_COLLOCATOR: Note, MODIS_DIR not defined'
        endif
     endif
;    Add MODIS to the list of instruments subsetted
     if n_elements(inst_list) eq 0 then inst_list = 'MODIS'            $
                                   else inst_list = [inst_list,'MODIS']
;    A list of the variables to read from the MOD04 files
     modvars = ['Latitude', 'Longitude', 'Scan_Start_Time',            $
                'Cloud_Mask_QA', 'Optical_Depth_Land_And_Ocean',       $
                'Image_Optical_Depth_Land_And_Ocean',                  $
                'Optical_Depth_Ratio_Small_Land_And_Ocean',            $
                'Aerosol_Type_Land', 'Fitting_Error_Land',             $
                'Angstrom_Exponent_Land', 'Mass_Concentration_Land',   $
                'Cloud_Fraction_Land', 'Quality_Assurance_Land',       $
                'Mass_Concentration_Ocean', 'Effective_Radius_Ocean',  $
                'Angstrom_Exponent_1_Ocean', 'Cloud_Fraction_Ocean',   $
                'Quality_Assurance_Ocean'                              ]
;    Get a file listing for the current date
     ddd  = calc_dayofyear(yr,mt,dy)
     yyyy = strtrim(yr,2)
     sfiles = file_search(modisdir, 'MOD04_L2.A'+yyyy+ddd+'*.hdf',     $
                          count=nfiles)
;    Structure used in concatinated subsetted MODIS data structures
;    below
     modexcep = { tags: ['QUALITY_ASSURANCE_LAND',                     $
                         'QUALITY_ASSURANCE_OCEAN'],                   $
                  dims: [1,1]                                          }
;    Read in each file, apply the spatial and temporal filters and pull
;    out the points which match the flight path.
;    Concatinate the resulting structures into one big one.
     if nfiles gt 0 then for i=0,nfiles-1 do begin
        mod04tmp = read_hdf4(sfiles[i], /quiet, names=modvars)
        if (size(mod04tmp))[0] eq 0 then begin
           message,/info,                                              $
                   'ADIENT_DATA_COLLOCATOR: Warning, failed to read '+ $
                   files[i]
        endif else begin
;          Extract all the data within the flight region and time, but 
;          don't yet test for the data along the actual flight path...
;          Convert MODIS Julian to the absolute scale
           modtim = double(mod04tmp.scan_start_time/86400.0) + $
                    julday(1,1,1993,0,0,0)
           inregion = where(mod04tmp.latitude gt min(faamlat)-2.0 and  $
                            mod04tmp.latitude lt max(faamlat)+2.0 and  $
                            mod04tmp.longitude gt min(faamlon)-2.0 and $
                            mod04tmp.longitude lt max(faamlon)+2.0 and $
                            modtim gt min(faamtim)-maxtime/1440. and   $
                            modtim lt max(faamtim)+maxtime/1440.,      $
                            nregion)

           if nregion gt 0 then begin
;             Add this file to the list of input files
              input_files=[input_files,sfiles[i]]
;             Extract the data within the region of interest from the
;             modis file.
              moddim = (size(mod04tmp.latitude))[1:2]
              mod04tmp = subset_struct(mod04tmp,inregion,moddim)
;             Manually do the arrays that have different dimensions
;             Quality_assurance_* arrays
              tmp1a = mod04tmp.quality_assurance_land
              tmp1b = mod04tmp.quality_assurance_ocean
              qal  = make_array(dimen=[5,n_elements(mod04tmp.latitude)],$
                                /float)
              qao  = qal
              for j=0,4 do begin
                 tmp2 = tmp1a[j,*,*]
                 qal[j,*] = tmp2[inregion]
                 tmp2 = tmp1b[j,*,*]
                 qao[j,*] = tmp2[inregion]
              endfor
;             Mass concentration, effective radius and Angstrom over
;             ocean
              tmp1a = mod04tmp.mass_concentration_ocean
              tmp1b = mod04tmp.effective_radius_ocean
              tmp1c = mod04tmp.angstrom_exponent_1_ocean
              mco = make_array(dimen=[n_elements(mod04tmp.latitude),2],$
                               /float)
              ero = mco
              aeo = mco
              for j=0,1 do begin
                 tmp2 = tmp1a[*,*,j]
                 mco[*,j] = tmp2[inregion]
                 tmp2 = tmp1b[*,*,j]
                 ero[*,j] = tmp2[inregion]
                 tmp2 = tmp1c[*,*,j]
                 aeo[*,j] = tmp2[inregion]
              endfor
;             Rebuild the fully subsetted mod04 structure
              mod04_1 = {NO_DATASETS : mod04tmp.NO_DATASETS}
              tags = tag_names(mod04tmp)
              for j=1,n_elements(tags)-1 do begin
                 if tags[j] eq 'QUALITY_ASSURANCE_LAND' then begin
                    mod04_1 = create_struct(mod04_1,tags[j],qal)
                 endif else                                            $
                 if tags[j] eq 'QUALITY_ASSURANCE_OCEAN' then begin
                    mod04_1 = create_struct(mod04_1,tags[j],qao)
                 endif else                                            $
                 if tags[j] eq 'MASS_CONCENTRATION_OCEAN' then begin
                    mod04_1 = create_struct(mod04_1,tags[j],mco)
                 endif else                                            $
                 if tags[j] eq 'EFFECTIVE_RADIUS_OCEAN' then begin
                    mod04_1 = create_struct(mod04_1,tags[j],ero)
                 endif else                                            $
                 if tags[j] eq 'ANGSTROM_EXPONENT_1_OCEAN' then begin
                    mod04_1 = create_struct(mod04_1,tags[j],aeo)
                 endif else begin
                    mod04_1 = create_struct(mod04_1,tags[j],mod04tmp.(j))
                 endelse
              endfor
              if (size(mod04))[0] eq 0 then mod04 = mod04_1            $
              else mod04 = concat_struct(mod04,mod04_1,excep=modexcep)
           endif
        endelse
     endfor
     if (size(mod04))[0] eq 0 then begin
        mod04 = -1
        message,/info,'No MODIS L2 data matches specified flight path'
        goto, nomod04
     endif
;    For each sample we have from the FAAM data, apply the selection
;    criteria to the subsetted MODIS data. This produces the structure
;    we return.
;    Convert the maxdist value into radians
     maxrad = maxdist / 6378.
;    Calculate Julian date for each subsetted MODIS point
     modtim = double(mod04.scan_start_time/86400.0) +                  $
              julday(1,1,1993,0,0,0)
     moddim = (size(mod04.latitude))[1]
     outdef = 0
     for i=0,nfaam-1 do begin
;       Calculate the angular distance between each point in the 
;       satellite swath and the aircraft location
        distrad = 2.0 * asin(                                          $
                  sqrt((sin(!dtor*(mod04.latitude-faamlat[i])/2))^2 +  $
                       cos(!dtor*mod04.latitude)*cos(!dtor*faamlat[i])* $
                       (sin(!dtor*(mod04.longitude-faamlon[i])/2))^2)  $
                            )
;       Apply the selection criteria
        ok = where(distrad le maxrad and $
                   abs(modtim-faamtim[i]) le maxtime/1440., nok)
;       Place in the output structure
        if nok gt 0 then begin
           outtmp = subset_struct(mod04,ok,moddim)
;          Again, we have to do some elements of the data structure by
;          hand
;          Quality_assurance_* arrays
           tmp1a = outtmp.quality_assurance_land
           tmp1b = outtmp.quality_assurance_ocean
           qal  = make_array(dimen=[5,n_elements(outtmp.latitude)],$
                             /float)
           qao  = qal
           for j=0,4 do begin
              tmp2 = tmp1a[j,*]
              qal[j,*] = tmp2[ok]
              tmp2 = tmp1b[j,*]
              qao[j,*] = tmp2[ok]
           endfor
;          Mass concentration, effective radius and Angstrom over
;          ocean
           tmp1a = outtmp.mass_concentration_ocean
           tmp1b = outtmp.effective_radius_ocean
           tmp1c = outtmp.angstrom_exponent_1_ocean
           mco = make_array(dimen=[n_elements(outtmp.latitude),2],$
                            /float)
           ero = mco
           aeo = mco
           for j=0,1 do begin
              tmp2 = tmp1a[*,j]
              mco[*,j] = tmp2[ok]
              tmp2 = tmp1b[*,j]
              ero[*,j] = tmp2[ok]
              tmp2 = tmp1c[*,j]
              aeo[*,j] = tmp2[ok]
           endfor
;          Rebuild the fully subsetted output structure
           modisl2_1 = {NO_DATASETS : outtmp.NO_DATASETS}
           tags = tag_names(outtmp)
           for j=1,n_elements(tags)-1 do begin
              if tags[j] eq 'QUALITY_ASSURANCE_LAND' then begin
                 modisl2_1 = create_struct(modisl2_1,tags[j],qal)
              endif else                                            $
                 if tags[j] eq 'QUALITY_ASSURANCE_OCEAN' then begin
                 modisl2_1 = create_struct(modisl2_1,tags[j],qao)
              endif else                                            $
                 if tags[j] eq 'MASS_CONCENTRATION_OCEAN' then begin
                 modisl2_1 = create_struct(modisl2_1,tags[j],mco)
              endif else                                            $
                 if tags[j] eq 'EFFECTIVE_RADIUS_OCEAN' then begin
                 modisl2_1 = create_struct(modisl2_1,tags[j],ero)
              endif else                                            $
                 if tags[j] eq 'ANGSTROM_EXPONENT_1_OCEAN' then begin
                 modisl2_1 = create_struct(modisl2_1,tags[j],aeo)
              endif else begin
                 modisl2_1 = create_struct(modisl2_1,tags[j],outtmp.(j))
              endelse
           endfor
        endif else modisl2_1 = {NO_DATASETS : 0}
        
        if outdef eq 0 and modisl2_1.no_datasets[0] gt 0 then begin
           outdef = 1
           modisl2 = { m1: modisl2_1}
        endif else if modisl2_1.no_datasets[0] gt 0 then begin
           outdef = outdef+1
           label = 'm'+strtrim(outdef,2)
           modisl2 = create_struct(modisl2,label,modisl2_1)
        endif
     endfor
     if outdef eq 0 then message,/info,'No matching MODIS L2 data found'
nomod04:
  endif ; END of MODIS code
; *********************************************************************

;                            * AATSR IOAP *

; If required, create a NCDF file of the output
  if keyword_set(save) then begin
     if save ne 1 then outfile = save else outfile = 0
     save_output_data, faam, faamidx, input_files, inst_list,          $
                       modisl2=modisl2, aatsr=aatsr, meris=meris,      $
                       seviri=seviri, misr=misr, outfile=outfile
  endif

end
