;+
; function read_hdf4, filename, /quiet, /info, names=names, $
;                     /global_attributes, /variable_attributes
; 
; Reads data from a HDF4 file into an IDL structure. Automatically
; applies the scaling factors and offsets to the data arrays often found
; in NASA HDF-EOS data.
;
; RETURN VALUE
; Usually returns a structure containing the data from the HDF file.
; However, if the /info keyword is set, then the value 1 will be
; returned on completion. If the file is not a valid HDF file, 0 will
; be returned.
;
; INPUT PARAMETER
; filename The name (and path) of the HDF file to read.
;
; KEYWORDS
; /quiet   Supresses information usually printed to the screen.
; /info    Rather than reading the file, the routine prints detailed
;          information about the file contents to the screen.
; names=names  Allows the user to specify a subset of variables to read
;          from the file (by default all data will be read). This must
;          be a string array, with each element containing a variable
;          name AS IT APPEARS IN THE HDF FILE. It is case sensitive!
;          To get a list of the variable names in a given file, try
;          using the /info keyword.
; /global_attributes If set, the routine will read the global, or file,
;          attributes from the file as well as the data. These will
;          appear as variables in the output structure.
; /variable_attributes If set, the routine will read the attributes of
;          each variable into the output structure as well as the data.
;          In this case each data variable within the returned
;          structure will itself be a structure (rather than a simple 
;          data array). The data will be stored in a variable called
;          DATA within this sub-structure.
;
; HISTORY
; 10/11/2008 G Thomas Original, hacked from "hdfread.pro" from the
;                     coyote library of IDL routines.
; 19/03/2009 G Thomas Bug fix: code now robustly deals with variable
;                     which lack any attributes
;-

FUNCTION TEST_ISHDF,filename
CATCH, err
IF (err EQ 0) THEN RETURN, HDF_ISHDF(filename) $
        ELSE RETURN, 0
END ;------------------------------------------------------------------

FUNCTION CHECK_LEGAL_NAME, inName
;     Check that inName doesn't contain any illegal characters
      tmpName = strsplit(inName,' ,.-+=/$%()[]{}<>',/extract)
      if n_elements(tmpName) gt 1 then outName = strjoin(tmpName,'_') $
                                  else outName = inName
      return, outName
end ;------------------------------------------------------------------


; =====================================================================
; BEGINNING OF MAIN FUNCTION
function read_hdf4, filename, quiet=quiet, info=info, $
                    names=names, global_attributes=global_attributes, $
                    variable_attributes=variable_attributes

; Open file and initialize the SDS interface.

IF N_ELEMENTS(filename) EQ 0 THEN filename = PICKFILE()
IF NOT TEST_ISHDF(filename) THEN BEGIN
   PRINT, 'Invalid HDF file ...'
   RETURN,-1
ENDIF ELSE $
   if ~keyword_set(quiet) then PRINT, 'Valid HDF file. Opening: ' + $
                                      filename

newFileID = HDF_SD_START(filename, /READ)

; If the info keyword is set, simply print information about the file
; and exit
HDF_SD_FILEINFO, newFileID, datasets, attributes
numPalettes = HDF_DFP_NPALS(filename)
if keyword_set(info) then begin
;  What is in the file. Print the number of datasets, attributes, and 
;  palettes.
   PRINT, 'Reading number of datasets and file attributes in file ...'
   PRINT, ''
   PRINT, 'No. of Datasets:   ', datasets
   PRINT, 'No. of File Attributes: ', attributes
   PRINT, 'No. of Palettes:   ', numPalettes

   ; Print the name of each file attribute.
   PRINT, ''
   PRINT, 'Printing name of each file attribute...'
   FOR j=0, attributes-1 DO BEGIN
      HDF_SD_ATTRINFO, newFileID, j, NAME=thisAttr
      PRINT, 'File Attribute No. ', + STRTRIM(j, 2), ': ', thisAttr
   ENDFOR

;  Print the name of each SDS and associated data attributes.

   PRINT, ''
   PRINT, 'Printing name of each data set and its associated data attributes...'
   FOR j=0, datasets-1 DO BEGIN
      thisSDS = HDF_SD_SELECT(newFileID, j)
      HDF_SD_GETINFO, thisSDS, NAME=thisSDSName, NATTS=numAttributes
      PRINT, 'Dataset No. ', STRTRIM(j,2), ': ', thisSDSName
      FOR k=0,numAttributes-1 DO BEGIN
         HDF_SD_ATTRINFO, thisSDS, k, NAME=thisAttrName
         PRINT, '   Data Attribute: ', thisAttrName
      ENDFOR
      PRINT, ''
   ENDFOR

;  Find the index of the "Gridded Data" SDS.

   index = HDF_SD_NAMETOINDEX(newFileID, "Gridded Data")
   
   if index ge 0 then begin
;     Select the Gridded Data SDS.

      thisSdsID = HDF_SD_SELECT(newFileID, index)
   
;     Print the names of the Gridded Data attributes.

      PRINT, ''
      PRINT, 'Printing names of each Gridded Data Attribute...'
      HDF_SD_GETINFO, thisSdsID, NATTS=numAttributes
      PRINT, 'Number of Gridded Data attributes: ', numAttributes
      FOR j=0,numAttributes-1 DO BEGIN
         HDF_SD_ATTRINFO, thisSdsID, j, NAME=thisAttr
         PRINT, 'SDS Attribute No. ', + STRTRIM(j, 2), ': ', thisAttr
      ENDFOR
   endif
;  Close file and return
   HDF_SD_END, newFileID
   return, 0
endif else begin
;  Info keyword wasn't set, actually read the data into a structure for
;  returning
   data  = create_struct( 'No_datasets',        datasets,   $
                          'No_glob_attributes', attributes, $
                          'No_Palettes',        numPalettes )
;  Read global attributes if they're required
   if keyword_set(global_attributes) then for j=0, attributes-1 do begin
      HDF_SD_ATTRINFO, newFileID, j, NAME=thisAttr, data=thisAttrDat
      outName = check_legal_name(thisAttr)
      data = create_struct(data, outName, thisAttrDat)
   endfor

;  Now move on to reading the SDSs
   for j=0, datasets-1 do begin
      thisSDS = HDF_SD_SELECT(newFileID, j)
      HDF_SD_GETINFO, thisSDS, NAME=thisSDSName, NATTS=numAttributes      
;     If the names keyword has been set, then check the current 
;     variable agains the list. If it's not on the list, skip it
      if keyword_set(names) then begin
         inlist = where(names eq thisSDSName)
         if inlist[0] lt 0 then goto, skipSDS
      endif
      outName = check_legal_name(thisSDSName)
      HDF_SD_GETDATA, thisSDS, thisDATA
;     Extract a list of attribute names. If the "scale_factor" and/or
;     "add_offset" are present, then change the data to a floating
;     point array and apply them.
      if numAttributes gt 0 then AttrNames = strarr(numAttributes) $
                            else AttrNames = ['']
      for k=0,numAttributes-1 do begin
         HDF_SD_ATTRINFO, thisSDS, k, NAME=thisAttrName
         AttrNames[k] = thisAttrName
      endfor
      i_scale  = (where(AttrNames eq 'scale_factor'))[0]
      i_offset = (where(AttrNames eq 'add_offset'))[0]
      i_fill   = (where(AttrNames eq '_FillValue'))[0]
      if (i_scale ge 0) or (i_offset ge 0) then begin
         if i_scale ge 0 then begin
            HDF_SD_ATTRINFO, thisSDS, i_scale, DATA=thisScale
            thisScale = thisScale[0]
         endif else thisScale = 1.0
         
         if i_offset ge 0 then begin
            HDF_SD_ATTRINFO, thisSDS, i_offset, DATA=thisOffset
            thisOffset = thisOffset[0]
         endif else thisOffset = 0.0
         tmp = float(thisDATA) * thisScale + thisOffset
;        Check for fill values and replace with the correct value
         if i_fill ge 0 then begin
            HDF_SD_ATTRINFO, thisSDS, i_fill, DATA=fillvalue
            isfill = where(thisDATA eq fillvalue)
            if isfill[0] ge 0 then tmp[isfill] = float(fillvalue)
         endif
         thisDATA = float(tmp)
      endif
;     if the variable_attributes keyword has been set, read all the
;     attributes and data into a sub structure, otherwise, just read
;     the data array itself
      if keyword_set(variable_attributes) and numAttributes gt 0 then $
         begin
         HDF_SD_ATTRINFO, thisSDS, 0, DATA=thisAttrDat
         datstr = create_struct(AttrNames[0], thisAttrDat)
         for k=1,numAttributes-1 do begin
            HDF_SD_ATTRINFO, thisSDS, k, DATA=thisAttrDat
;           If the data has been scaled and the certain attributes are
;           present, they should be scaled as well
            if ((i_scale ge 0) or (i_offset ge 0)) and $
               (AttrNames[k] eq 'valid_range') then begin
               thisAttrDat = float(thisAttrDat) * thisScale + thisOffset
            endif
            outAttrName = check_legal_name(AttrNames[k])
;           Data is always returned as an array, even when it's one
;           value. Tidy it up...
            if n_elements(thisAttrDat) eq 1 then $
               thisAttrDat = thisAttrDat[0]
            datstr = create_struct(datstr, outAttrName, thisAttrDat)               
         endfor
;        Add in the data itself
         datstr = create_struct(datstr, 'Data', thisDATA)
;        Add this into the output
         data = create_struct(data, outName, datstr)
      endif else begin
;     If the variable attributes aren't required, simply put the array
;     of data into the output structure
         data = create_struct(data, outName, thisDATA)
      endelse
      skipSDS: ; Landing point if the current SDS isn't required
   endfor
   HDF_SD_END, newFileID
   return,data
endelse
END ; =================================================================
