from netCDF4 import Dataset

class L1b_Hiros:
  """ Data and Methods for HIROS L1B structure 

  Version
    28JUN24 AD Change 'title' to 'Title' etc for creating netCDF glob.atts.
    14JUN24 AD Change Noise from (nmic,nalt) to (nmic,nmax)
    05APR24 AD Adapted from IDL procedures

  Usage
    from l1b_hiros_class import L1b_Hiros

    l1b = L1b_Hiros ( l1bfil )  # load existing netCDF L1B file
    l1b = L1b_Hiros ( )         # return empty L1B data object
    l1b.write ( l1bfil )        # Write L1B data object to netCDF file

  Attributes
    see modules

  Public Methods
    write    : Write HIROS L1B data structure to netCDF file

  Private Methods
    __init__ : Initialisation returns L1B structure
    _null    : Make empty L1B data structure
    _read    : Read L1B HIROS netCDF file and return as data structure
  """

  def __init__ ( self, l1bfil=None ): 
    """ Initialise 

    Parameters
      l1bfil str : name of netCDF L1B file to be read (optional)

    Returns
      l1b object
    """

    if l1bfil:
      self._read ( l1bfil )
    else:
      self._null ( )

  def _null ( self ):
    """  Make empty L1B data structure  """

    self.title    = None
    self.created  = None
    self.source   = None
    # Array dimensions
    self.nalt = None
    self.nmic = None
    self.nmax = None
    # Scalars
    self.satellite    = None
    self.instrument   = None
    self.orbit        = None
    self.sunrise      = None
    # Arrays dimensioned nmic
    self.mic_lab      = None
    self.mic_npt      = None
    self.mic_min      = None
    self.mic_max      = None
    self.mic_res      = None
    # Arrays dimensioned nalt
    self.julian_day   = None
    self.milliseconds = None
    self.altitude     = None
    self.latitude     = None
    self.longitude    = None
    self.rad_curve    = None
    # Arrays dimensioned nmic,nalt
    self.noise        = None
    self.alt_offset   = None
    self.alt_trend    = None
    self.alt_quad     = None
    self.quality      = None
    # Array dimensioned nmic,nalt,npt
    self.transmittance = None

  def _read ( self, l1bfil ): 
    """ Read HIROS L1B structure from netCDF file
  
    Parameters
      l1bfil str : Name of netCDF file
    """

    nc = Dataset ( l1bfil )
    # Global attributes
    self.title    = nc.getncattr('Title')
    self.created  = nc.getncattr('Created')
    self.source   = nc.getncattr('Source')
    # Array dimensions
    self.nalt = len ( nc.dimensions['NAlt'] )
    self.nmic = len ( nc.dimensions['NMic'] )
    self.nmax = len ( nc.dimensions['NMax'] )
    # Scalars
    self.satellite    = nc['Satellite'][0]
    self.instrument   = nc['Instrument'][0]
    self.orbit        = nc['Orbit'][0]
    self.sunrise      = nc['Sunrise'][0]
    # Arrays dimensioned nmic
    self.mic_lab      = nc['Mic_Lab'][:]   # size nmic
    self.mic_npt      = nc['Mic_Npt'][:]
    self.mic_min      = nc['Mic_Min'][:]
    self.mic_max      = nc['Mic_Max'][:]
    self.mic_res      = nc['Mic_Res'][:]
    # Arrays dimensioned nalt
    self.julian_day   = nc['Julian_Day'][:]  # size nalt
    self.milliseconds = nc['Milliseconds'][:]
    self.altitude     = nc['Altitude'][:]
    self.latitude     = nc['Latitude'][:]
    self.longitude    = nc['Longitude'][:]
    self.rad_curve    = nc['Rad_Curve'][:]
    # Arrays dimensioned nmic,nalt
    self.noise        = nc['Noise'][:,:]  # size nmic,nmax
    self.alt_offset   = nc['Alt_Offset'][:,:]
    self.alt_trend    = nc['Alt_Trend'][:,:]
    self.alt_quad     = nc['Alt_Quad'][:,:]
    self.quality      = nc['Quality'][:,:]
    # Arrays dimensioned nmic,nmax
    self.noise        = nc['Noise'][:,:]  # size nmic,nmax
    # Array dimensioned nmic,nalt,nmax
    self.transmittance = nc['Transmittance'][:,:,:]

  def write ( self, l1bfil ):
    """ Write HIROS L1B structure to netCDF file

    Parameters
      l1bfil str : netCDF filename

    Returns
      None
    """

    # Open output file
    nc = Dataset ( l1bfil, mode='w' ) # don't use NETCDF4_CLASSIC' so can use 'str'

    # Define Global attributes
    nc.Title   = self.title
    nc.Created = self.created
    nc.Source  = self.source

    # Define Array dimensions
    nc.createDimension('NMic',self.nmic)
    nc.createDimension('NAlt',self.nalt)
    nc.createDimension('NMax',self.nmax)

    # Scalar Variables
    sat_ncv = nc.createVariable ( 'Satellite', str )
    sat_ncv[0] = self.satellite

    ins_ncv = nc.createVariable ( 'Instrument', str )
    ins_ncv[0] = self.instrument

    orb_ncv = nc.createVariable ( 'Orbit', int )
    orb_ncv[0] = self.orbit

    sun_ncv = nc.createVariable ( 'Sunrise', 'i1' )
    sun_ncv[0] = self.sunrise

    # Assign nmic-sized variables
    mic_npt_ncv = nc.createVariable ( 'Mic_Npt', int, 'NMic' ) 
    mic_npt_ncv[:] = self.mic_npt

    mic_lab_ncv = nc.createVariable ( 'Mic_Lab', str, 'NMic' )
    for imic in range ( self.nmic ):
      mic_lab_ncv[imic] = self.mic_lab[imic]

    mic_min_ncv = nc.createVariable ( 'Mic_Min', float, 'NMic' )
    mic_min_ncv[:] = self.mic_min
    setattr ( mic_min_ncv, 'units', 'cm-1' )

    mic_max_ncv = nc.createVariable ( 'Mic_Max', float, 'NMic' )
    mic_max_ncv[:] = self.mic_max
    setattr ( mic_max_ncv, 'units', 'cm-1' )

    mic_res_ncv = nc.createVariable ( 'Mic_Res', float, 'NMic' )
    mic_res_ncv[:] = self.mic_res
    setattr ( mic_res_ncv, 'units', 'cm-1' )

    # Assign nalt-sized variables
    day_ncv = nc.createVariable ( 'Julian_Day', int, 'NAlt' )
    day_ncv[:] = self.julian_day
    setattr ( day_ncv, 'long_name', 'Day since 1st Jan 2000' )

    msc_ncv = nc.createVariable ( 'Milliseconds', int, 'NAlt' )
    msc_ncv[:] = self.milliseconds
    setattr ( msc_ncv, 'long_name', 'Milliseconds since midnight' )
    setattr ( msc_ncv, 'units', 'ms' )

    alt_ncv = nc.createVariable ( 'Altitude', float, 'NAlt' )
    alt_ncv[:] = self.altitude
    setattr ( alt_ncv, 'long_name', 'Geometric tangent point altitude' )
    setattr ( alt_ncv, 'units', 'km' )

    lat_ncv = nc.createVariable ( 'Latitude', float, 'NAlt' )
    lat_ncv[:] = self.latitude
    setattr ( lat_ncv, 'long_name', 'Tangent point latitude' )
    setattr ( lat_ncv, 'units', 'deg N' )

    lon_ncv = nc.createVariable ( 'Longitude', float, 'NAlt' )
    lon_ncv[:] = self.longitude
    setattr ( lon_ncv, 'long_name', 'Tangent point longitude' )
    setattr ( lon_ncv, 'units', 'deg E' )

    crv_ncv = nc.createVariable ( 'Rad_Curve', float, 'NAlt' )
    crv_ncv[:] = self.rad_curve
    setattr ( crv_ncv, 'long_name', 'Earth radius of curvature in LOS plane' )
    setattr ( crv_ncv, 'units', 'km' )

    # Assign (nmic,nalt)-sized variables
    qal_ncv = nc.createVariable ( 'Quality', int, ('NMic','NAlt') )
    qal_ncv[:,:] = self.quality

    off_ncv = nc.createVariable ( 'Alt_Offset', float, ('NMic','NAlt') )
    off_ncv[:,:] = self.alt_offset
    setattr ( off_ncv, 'long_name', 'Microwindow altitude offset' )
    setattr ( off_ncv, 'units', 'km' )

    lin_ncv = nc.createVariable ( 'Alt_Trend', float, ('NMic','NAlt') )
    lin_ncv[:,:] = self.alt_trend
    setattr ( lin_ncv, 'long_name', 'Tangent altitude linear trend' )
    setattr ( lin_ncv, 'units', 'km' )

    qad_ncv = nc.createVariable ( 'Alt_Quad', float, ('NMic','NAlt') )
    qad_ncv[:,:] = self.alt_quad
    setattr ( qad_ncv, 'long_name', 'Tangent altitude quadratic trend' )
    setattr ( qad_ncv, 'units', 'km' )

    # Assign (nmic,nmax)-sized variables
    noi_ncv = nc.createVariable ( 'Noise', float, ('NMic','NMax') )
    noi_ncv[:,:] = self.noise

    # Assign space for (nmic,nalt,nmax)-sized data
    tra_ncv = nc.createVariable ( 'Transmittance', float, ('NMic','NAlt','NMax') )
    tra_ncv[:,:,:] = self.transmittance
    nc.close()

#--------
