from netCDF4 import Dataset

class L1b_Hsdi:
  """ Attributes and methods for HSDI L1B structure

  Version
    05APR24 AD Original. Based on IDL software

  Usage
    from l1b_hsdi_class import L1b_Hsdi

    l1b = L1b_Hsdi ( inpfil )  # load existing netCDF L1B file
    l1b = L1b_Hsdi ( )         # return empty L1B data object    
    l1b.write ( outfil )       # Write L1B data object to netCDF file

  Attributes
    see modules

  Private Methods
    __init__ : Initialisation returns L1B structure
    _null    : Make empty L1B data structure 
    _read    : Read L1B HSDI netCDF file and return as data structure

  Public Methods
    write    : Write HSDI L1B data structure to netCDF file
  """
  
  def __init__ ( self, l1bfil=None ):
    """ Initialisation returns L1B structure

    Parameters
      l1bfil str : (Optional) Name of netCDF L1B file

    Description
      Either read an existing L1B netCDF file or just return an empty
      data object.
    """

    if l1bfil: 
      self._read ( l1bfil )
    else: 
      self._null ( )
    return

  def _null ( self ):
    """  Make empty L1B data structure  """

    self.title   = None 
    self.created = None
    self.source  = None
    # Array dimensions
    self.nimg = None
    self.nchn = None
    self.nmos = None
    self.ndat = None
    # Scalars
    self.satellite  = None
    self.instrument = None
    self.orbit      = None
    self.sunrise    = None
    # Arrays dimensioned nchn
    self.chn_lab    = None
    self.chn_x      = None
    self.chn_y      = None
    self.chn_alt    = None
    # Arrays dimensioned nmos
    self.mos_x      = None
    self.mos_y      = None
    self.mos_alt    = None
    # Arrays dimensioned nimg
    self.julian_day    = None
    self.milliseconds  = None
    self.altitude      = None
    self.latitude      = None 
    self.longitude     = None
    self.rad_curve     = None
    self.nuse          = None
    # Array dimensioned ndat
    self.idx_mos       = None
    # Arrays dimensioned ndat,nchn
    self.quality       = None
    self.noise         = None
    self.transmittance = None

  def _read ( self, l1bfil ):
    """ Read L1B HSDI netCDF file and return as data structure

    Parameters
      l1bfil  str : Name of HSDI L1B netCDF file

    Returns
      L1B as object - see code for attributes
    """

    nc = Dataset ( l1bfil ) 
    # Global attributes
    self.title    = nc.getncattr('Title')
    self.created  = nc.getncattr('Created')
    self.source   = nc.getncattr('Source')
    # Array dimensions
    self.nimg = len ( nc.dimensions['NImg'] )
    self.nchn = len ( nc.dimensions['NChn'] )
    self.nmos = len ( nc.dimensions['NMos'] )
    self.ndat = len ( nc.dimensions['NDat'] )
    # Scalars
    self.satellite     = nc['Satellite'][0]
    self.instrument    = nc['Instrument'][0]
    self.orbit         = nc['Orbit'][0]
    self.sunrise       = nc['Sunrise'][0]
    # Arrays dimensioned nchn
    self.chn_lab       = nc['Chn_Lab'][:]
    self.chn_x         = nc['Chn_X'][:]
    self.chn_y         = nc['Chn_Y'][:]
    self.chn_alt       = nc['Chn_Alt'][:]
    # Arrays dimensioned nmos
    self.mos_x         = nc['Mos_X'][:]
    self.mos_y         = nc['Mos_Y'][:]
    self.mos_alt       = nc['Mos_Alt'][:]
    # Arrays dimensioned nimg
    self.julian_day    = nc['Julian_Day'][:]  
    self.milliseconds  = nc['Milliseconds'][:] 
    self.altitude      = nc['Altitude'][:]
    self.latitude      = nc['Latitude'][:]
    self.longitude     = nc['Longitude'][:]
    self.rad_curve     = nc['Rad_Curve'][:]
    self.nuse          = nc['NUse'][:]
    # Array dimensioned ndat
    self.idx_mos       = nc['Idx_Mos'][:]  
    # Arrays dimensioned ndat,nchn
    self.quality       = nc['Quality'][:,:]
    self.noise         = nc['Noise'][:,:]
    self.transmittance = nc['Transmittance'][:,:]

  def write ( self, l1bfil ):
    """ Write HSDI L1B data structure to netCDF file

    Parameters
      l1bfil  str : Name of HSDI L1B netCDF file

    Returns
      None
    """

    # Open output file 
    nc = Dataset ( l1bfil, mode='w' ) 

    # Define Global attributes
    nc.Title   = self.title 
    nc.Created = self.created
    nc.Source  = self.source

    # Define Array dimensions
    nc.createDimension('NImg',self.nimg)
    nc.createDimension('NChn',self.nchn)
    nc.createDimension('NMos',self.nmos)
    nc.createDimension('NDat',self.ndat)

    # Scalar Variables
    sat_ncv = nc.createVariable ( 'Satellite', str )  
    sat_ncv[0] = self.satellite

    ins_ncv = nc.createVariable ( 'Instrument', str ) 
    ins_ncv[0] = self.instrument

    orb_ncv = nc.createVariable ( 'Orbit', int )
    orb_ncv[0] = self.orbit

    sun_ncv = nc.createVariable ( 'Sunrise', 'i1' )
    sun_ncv[0] = self.sunrise

    # Arrays dimensioned nchn
    chn_lab_ncv = nc.createVariable ( 'Chn_Lab', str, 'NChn' ) 
    for ichn in range(self.nchn):
      chn_lab_ncv[ichn] = self.chn_lab[ichn]

    chn_x_ncv = nc.createVariable ( 'Chn_X', int, 'NChn' ) 
    chn_x_ncv[:] = self.chn_x

    chn_y_ncv = nc.createVariable ( 'Chn_Y', int, 'NChn' ) 
    chn_y_ncv[:] = self.chn_y

    chn_alt_ncv = nc.createVariable ( 'Chn_Alt', float, 'NChn' ) 
    chn_alt_ncv[:] = self.chn_alt

    # Arrays dimensioned nmos
    mos_x_ncv   = nc.createVariable ( 'Mos_X', int, 'NMos' ) 
    mos_x_ncv[:] = self.mos_x

    mos_y_ncv   = nc.createVariable ( 'Mos_Y', int, 'NMos' ) 
    mos_y_ncv[:] = self.mos_y

    mos_alt   = nc.createVariable ( 'Mos_Alt', float, 'NMos' ) 
    mos_alt[:] = self.mos_alt

    # Arrays dimensioned nimg
    jdat_ncv  = nc.createVariable ( 'Julian_Day', int, 'NImg' )  
    jdat_ncv[:] = self.julian_day

    msc_ncv = nc.createVariable ( 'Milliseconds', int, 'NImg' )  
    msc_ncv[:] = self.milliseconds  

    alt_ncv = nc.createVariable ( 'Altitude', float, 'NImg' )  
    alt_ncv[:] = self.altitude

    lat_ncv = nc.createVariable ( 'Latitude', float, 'NImg' )  
    lat_ncv[:] = self.latitude

    lon_ncv = nc.createVariable ( 'Longitude', float, 'NImg' )  
    lon_ncv[:] = self.longitude

    rad_ncv = nc.createVariable ( 'Rad_Curve', float, 'NImg' )  
    rad_ncv[:] = self.rad_curve

    nus_ncv = nc.createVariable ( 'Nuse', int, 'NImg' )  
    nus_ncv[:] = self.nuse

    # Array dimensioned ndat
    idm_ncv = nc.createVariable ( 'Idx_Mos', int, 'NDat' )  
    idm_ncv[:] = self.idx_mos

    # Arrays dimensioned nchn,ndat
    qal_ncv = nc.createVariable ( 'Quality', int, ('NDat', 'NChn') )
    qal_ncv[:,:] = self.quality

    noi_ncv = nc.createVariable ( 'Noise', float, ('NDat', 'NChn') )
    noi_ncv[:,:] = self.noise

    tra_ncv = nc.createVariable ( 'Transmittance', float, ('NDat', 'NChn') )
    tra_ncv[:,:] = self.transmittance

    nc.close()
#-------




