import numpy as np
from l1b_hiros_class import L1b_Hiros
from datetime import datetime

def hms ( ms ):
  ihr = int( ms // 3600000 )
  imin = int ( ms % 3600000 ) // 60000
  isec = round ( ( ms % 60000 ) // 1000 )
  return f'{ihr:02}{imin:02}{isec:02}'

def ymd ( jday ):
  jday0 = datetime(2000,1,1).toordinal()
  return datetime.fromordinal(jday+jday0).strftime('%Y%m%d')

def l1c_hiros ( l1bfil='hiros.l1b', l1cfil='hiros.l1c', 
                altmin=None, altmax=None ):
  """ Cubemap L2 Preprocesor: convert HIROS L1B data to MORSE L1C file

  Version  (change VERSID to match this)
    14JUN24 AD Add noise as a spectrum rather than a scalar
    22FEB24 AD Original. Derived from IDL procedure l1c_hiros.pro

  Parameters
    l1bfil str : Name of input L1B file (netCDF)
    l1cfil str : Name of output L1C file (text)
    altmin flt : Min.tangent altitude [km] for inclusing in L1C file
    altmax flt : Max.tangent altitude [km] for inclusing in L1C file

  Usage
    eg: from l1c_hiros import l1c_hiros

        l1c_hiros ( 'l1b_hiros.nc', 'hiros.l1c' )
  """

  VERSID = '14JUN24'

  l1b  = L1b_Hiros ( l1bfil ) 
  nalt = l1b.nalt
  nmic = l1b.nmic
  
  # Check that all microwindows have same resolution
  resln = l1b.mic_res[0]
  for imic in range(1,nmic):
    if l1b.mic_res[imic] != resln:
      print('F-l1c_hiros:  Microwindows have different spectral resolutions')
      exit()

  with open ( l1cfil, 'w' ) as f:
    sp = ''  # dummy print variable for adding blank spaces
    f.write ( '! MORSE l1c file created by l1c_hiros.py v' + VERSID + '\n' )

    f.write ( f'{3.4:10.1f}{sp:25}(=format id)\n' ) 

    f.write ( f'{2:10}{resln:10.4f}{sp:15}(2=limb tra, resln)\n' )

    f.write ( f'{l1b.instrument:>10}{l1b.satellite:>10}{sp:15}' + \
               '(=Instrument,satellite)\n' )

    # Establish various time/date forms
    jday = l1b.julian_day[0]
    f.write ( f'{ymd(jday):>10}{jday:>10}{sp:15}' \
              '(=nominal date, day# since 1Jan2000)\n' )

    orb = f'{l1b.orbit:05}'    # fill out to 5 chars with leading zeros
    hmssta = hms ( l1b.milliseconds[0] )
    hmsend = hms ( l1b.milliseconds[nalt-1] )
    f.write ( f'{orb:>10}{hmssta:>10}{hmsend:>10}{sp:5}' + \
               '(=Orbit#, Event Start, End HHMMSS)\n') 

    nscn = 1         # No. profiles in file
    f.write ( f'{nscn:10}{sp:25}(=No.scans)\n' )

    alt = np.array ( l1b.altitude[:] )
    if altmin is None: altmin = np.min(alt) - 1
    if altmax is None: altmax = np.max(alt) + 1
    idxalt = np.nonzero( (alt >= altmin) & (alt <= altmax) )[0]
    nidx = len(idxalt)
    f.write ( f'{nidx:10}{"GEO":>10}{sp:15}' +
               '(=No.sweeps per scan, then nominal alts [km])\n' )

    # In L1B alts are in time order, but L1C always high to low
    if l1b.sunrise: idxalt = idxalt[::-1]

    if ( alt[idxalt[0]] < alt[idxalt[nidx-1]] ):
      print('F-l1c_hiros: L1B altitude sequence does not match SUNRISE flag')
      exit()
       
    # Write list of tangent altitudes
    ifld = 0
    for idx in idxalt:
      f.write ( f'{alt[idx]:8.3f}' )
      ifld += 1 
      if ifld % 10 == 0: f.write ('\n')
    if ifld % 10 != 0: f.write ('\n')    # ensure <CR> after last field

    # Single scan in HIROS file
    iscn = 1
    f.write( f'{iscn:10}{sp:5}(=Scan#)\n' )

    # Fixed values for these fields
    LST = 0.0
    SZA = 90.0
    CLDRAD = 0.0
    CLDIDX = 0.0

    # L1B contains linear altitude trend from low freq to high freq but this
    # remains the same even if sunrise data sequence is reversed
    for idx in idxalt:
      f.write ( '!yyyymmdd hhmmss  millisec Scn Swp   Lat     Lon' + 
                    '   LST     SZA   CldRad CldIdx\n' )
      ymdalt = ymd ( l1b.julian_day[idx] ) 
      f.write ( f'{ymdalt:>9}' )
      msec = l1b.milliseconds[idx]
      hmsalt = hms(msec) 
      lat = l1b.latitude[idx] 
      lon = l1b.longitude[idx] 
      f.write ( f'{hmsalt:>7}' )
      f.write ( f'{msec:10}' )
      f.write ( f'{iscn:4}' )
      f.write ( f'{idx+1:4}' )
      f.write ( f'{lat:7.2f}' )
      f.write ( f'{lon:8.2f}' )
      f.write ( f'{LST:7.2f}' )
      f.write ( f'{SZA:7.2f}' )
      f.write ( f'{CLDRAD:8.1f}' )
      f.write ( f'{CLDIDX:7.2f}' )
      f.write ( '\n' )

      alt = l1b.altitude[idx]
      rad = l1b.rad_curve[idx]
      f.write ( '!NMW GeoNom  GeoSwp  RadCrv\n' )
      f.write ( f'{nmic:3}{alt:8.3f}{alt:8.3f}{rad:9.3f}\n' )

      for imic in range(nmic):
        f.write ( '! Label   NPts      WnoMin      WnoMax     Noise' + \
                    '    Offset     Drift Quadratic\n' )
        f.write ( f'{l1b.mic_lab[imic]:<8}' )
        f.write ( f'{l1b.mic_npt[imic]:6}' )
        f.write ( f'{l1b.mic_min[imic]:12.4f}' )
        f.write ( f'{l1b.mic_max[imic]:12.4f}' )
        f.write ( f'{-1.0:10.4f}' )       # -1 flags Noise will be a spectrum
        f.write ( f'{l1b.alt_offset[imic,idx]:10.3f}' )
        f.write ( f'{l1b.alt_trend[imic,idx]:10.3f}' )
        f.write ( f'{l1b.alt_quad[imic,idx]:10.3f}\n' )

        for ipt in range ( l1b.mic_npt[imic] ):
          f.write ( f'{l1b.transmittance[imic,idx,ipt]:10.6f}' )
          if ipt % 8 == 7: f.write ( '\n' )
        if ipt % 8 != 7: f.write ( '\n' )

        for ipt in range ( l1b.mic_npt[imic] ):
          f.write ( f'{l1b.noise[imic,ipt]:10.6f}' )
          if ipt % 8 == 7: f.write ( '\n' )
        if ipt % 8 != 7: f.write ( '\n' )
#----

#l1c_hiros ( 'l1b_hiros.nc', 'hiros.l1c', altmin=5, altmax=60 )
#l1c_hiros ( 'l1b_hiros_static.nc', 'hiros_static.l1c', altmin=5, altmax=60 )
l1c_hiros ( 'l1b_hiros_nofov.nc', 'hiros_nofov.l1c', altmin=5, altmax=60 )


