PRO L1C_HSDI, L1BFIL, L1CFIL, CHNLST=CHNLST, ALTMIN=ALTMIN, ALTMAX=ALTMAX
;
; VERSION
    VERSID = '29AUG23' ; Format v3.3, add GEO grid type, extra MW header record
;   VERSID = '22APR22' ; Use noise from l1b file rather than local value
;   VERSID = '18JAN22' ; Renamed from make_l1c_hsdi to l1c_hsdi
;   VERSID = '17JAN22' ; Change RAD_CURV to RAD_CURVE
;   VERSID = '19AUG21' ; Add CHNLST, ALTMIN, ALTMAX optional arguments. Allow for NIMG=1
;   VERSID = '17AUG21' ; Correction: IOFF calculation
;   VERSID = '09AUG21' ; Left-align channel label
;   VERSID = '02AUG21' ; Comment changes only
;   VERSID = '22JUL21' ; Original
;
; DESCRIPTION
;   L2 Preprocessor to convert HSDI L1B to L1C, suitable for MORSE input.
;
; ARGUMENTS
;   L1BFIL String  I  Name of input L1B file
;   L1CFIL String  I  Name of output L1C file
;
; EXTERNAL
;   Fnc READ_L1B_HSDI  Read L1B HSDI data
;   Fnc HMS             Return string HHMMSS for millisecs within day
;   Fnc YMD             Return string YYYYMMDD for Julian day#
;
; USAGE
;  eg: L1C_HDSI, 'l1b_hsdi.nc', 'hsdi.l1c' 

L1B = READ_L1B_HSDI ( L1BFIL ) 
NIMG = L1B.NIMG
NCHN = L1B.NCHN

IF NOT KEYWORD_SET ( CHNLST ) THEN CHNLST = L1B.CHN_LAB
IF NOT KEYWORD_SET ( ALTMIN ) THEN ALTMIN = -999.0
IF NOT KEYWORD_SET ( ALTMAX ) THEN ALTMAX = 999.0

OPENW, LUNL1C, L1CFIL, /GET_LUN
PRINTF, LUNL1C, '! MORSE l1c file created by l1c_hsdi.pro v' + VERSID
PRINTF, LUNL1C, '       3.3                         (=format id)'
PRINTF, LUNL1C, 2, 0.0, '(2=limb tra, resln)', FORMAT='(I10,F10.4,15X,A)'
PRINTF, LUNL1C, L1B.INSTRUMENT, L1B.SATELLITE, '(=Instrument,satellite)', $
  FORMAT='(2A10,15X,A)'

; Establish various time/date forms
JDAY = L1B.JULIAN_DAY[0]
PRINTF, LUNL1C, YMD(JDAY), JDAY, '(=nominal date, day# since 1Jan2000)', $
  FORMAT='(A10,I10,15X,A)'

PRINTF, LUNL1C, L1B.ORBIT, HMS(L1B.MILLISECONDS[0]), $
                           HMS(L1B.MILLISECONDS[NIMG-1]), $
 '(=Orbit#, Orbit Start, End HHMMSS)', FORMAT='(5X,I5.5,A10,A10,5X,A)'

NSCN = 1             ; No. of profiles in file
PRINTF, LUNL1C, NSCN, '(=No.scans)', FORMAT='(I10,25X,A)'

PRINTF, LUNL1C, NIMG, 'GEO', '(=No.sweeps per scan, then nominal alts [km])', $
  FORMAT='(I10,A10,15X,A)'

; In L1C altitudes are always high to low
IF L1B.SUNRISE THEN BEGIN
  I = NIMG-1
  J = 0
  K = -1
  PRINTF, LUNL1C, REVERSE(L1B.ALTITUDE), FORMAT='(10F8.3)'
ENDIF ELSE BEGIN
  I = 0
  J = NIMG-1
  K = 1
  PRINTF, LUNL1C, L1B.ALTITUDE, FORMAT='(10F8.3)'
ENDELSE

IF NIMG GT 1 AND L1B.ALTITUDE[I] LE L1B.ALTITUDE[J] THEN MESSAGE, $
  'F-L1C_HSDI: L1B altitude sequence does not match SUNRISE flag'

ISCN = 1
PRINTF, LUNL1C, ISCN, '(=Scan#)', FORMAT='(I10,5X,A)'

; Values in L1C file not defined/used in retrieval
LST = 0.0        ; local solar time
SZA = 90.0       ; solar zenith angle
CLDRAD = 0.0     ; cloud channel radiance
CLDIDX = 0.0     ; cloud index

FOR IIMG = I, J, K DO BEGIN
  PRINTF, LUNL1C, '!yyyymmdd hhmmss  millisec Scn Swp   Lat     Lon' + $
                    '   LST     SZA   CldRad CldIdx'
  ALT = L1B.ALTITUDE[IIMG]
  PRINTF, LUNL1C, ' ' + YMD ( L1B.JULIAN_DAY[IIMG] ) + ' ' + $
                  HMS ( L1B.MILLISECONDS[IIMG] ) + $
                  STRING ( L1B.MILLISECONDS[IIMG], FORMAT='(I10)' ) + $
                  STRING ( ISCN, FORMAT='(I4)' ) + $
                  STRING ( IIMG+1, FORMAT='(I4)' ) + $
                  STRING ( L1B.LATITUDE[IIMG], FORMAT='(F7.2)' ) + $
                  STRING ( L1B.LONGITUDE[IIMG], FORMAT='(F8.2)' ) + $
                  STRING ( LST, FORMAT='(F7.3)' ) + $
                  STRING ( SZA, FORMAT='(F7.2)' ) + $
                  STRING ( CLDRAD, FORMAT='(F8.1)' ) + $
                  STRING ( CLDIDX, FORMAT='(F7.2)' ) 
; Construct list of L1B measurements to be listed in L1C file
  RECLST = []
  IF IIMG EQ 0 THEN IOFF = 0 ELSE IOFF = TOTAL ( L1B.NUSE[0:IIMG-1] )
  FOR ICHN = 0, NCHN-1 DO BEGIN
    CHN_LAB = L1B.CHN_LAB[ICHN]
    IDUM = WHERE ( CHN_LAB EQ CHNLST, USECHN ) 
    IF NOT USECHN THEN CONTINUE
    FOR IDAT = IOFF, IOFF + L1B.NUSE[IIMG]-1 DO BEGIN
      IMOS = L1B.IDX_MOS[IDAT]
      ALTREL = L1B.CHN_ALT[ICHN] + L1B.MOS_ALT[IMOS]
      IF ( ALT + ALTREL GT ALTMAX ) OR ( ALT + ALTREL LT ALTMIN ) THEN CONTINUE
      REC = STRING ( CHN_LAB, ALTREL, L1B.TRANSMITTANCE[ICHN,IDAT], L1B.NOISE[ICHN,IDAT], $
        L1B.MOS_X[IMOS], L1B.MOS_Y[IMOS], FORMAT='(A-8,F10.3,F10.6,F10.6,I6,I6)' )
      RECLST = [ RECLST, REC ]
    ENDFOR
  ENDFOR
  PRINTF, LUNL1C, '!NMeas AltNOM  AltImg   RadCrv'
  PRINTF, LUNL1C, STRING ( N_ELEMENTS(RECLST), FORMAT='(I5)' ) + $
                  STRING ( ALT, FORMAT='(F8.3)' ) + $
                  STRING ( ALT, FORMAT='(F8.3)' ) + $
                  STRING ( L1B.RAD_CURVE[IIMG], FORMAT='(F9.3)' ) 
  IF N_ELEMENTS ( RECLST ) GT 0 THEN $
    PRINTF, LUNL1C, '! Chn_ID   Alt_Rel   Transm.     Noise Mos_X Mos_Y'
  FOR IREC = 0, N_ELEMENTS(RECLST)-1 DO PRINTF, LUNL1C, RECLST[IREC]
ENDFOR

FREE_LUN, LUNL1C

END

l1c_hsdi, 'l1b_hsdi.nc', 'hsdi.l1c', $
;   CHNLST=['HSDI_02', 'HSDI_05', 'HSDI_06','HSDI_07', 'HSDI_08'], $
           ALTMIN=5, ALTMAX=60


end
