import numpy as np
from l1b_hsdi_class import L1b_Hsdi
from datetime import datetime

def hms ( ms ):
  ihr = int( ms // 3600000 )
  imin = int ( ms % 3600000 ) // 60000
  isec = round ( ( ms % 60000 ) // 1000 )
  return f'{ihr:02}{imin:02}{isec:02}'

def ymd ( jday ):
  jday0 = datetime(2000,1,1).toordinal()
  return datetime.fromordinal(jday+jday0).strftime('%Y%m%d')

def l1c_hsdi ( l1bfil='hsdi.l1b', l1cfil='hsdi.l1c', 
                chnlst=None, altmin=None, altmax=None ):
  """ Cubemap L2 Preprocessor: convert HSDI L1B data to MORSE L1C file

  Version  (change VERSID to match this)
    03MAY24 AD Original. Derived from IDL procedure l1c_hsdi.pro

  Parameters
    l1bfil  str  : Name of input L1B file (netCDF)
    l1cfil  str  : Name of output L1C file (text)
    chnlst [str] : List of channels to include (else: use all)
    altmin  flt  : Minimum tangent altitude to include (else: use all)
    altmax  flt  : Maximum tangent altitude to include (else: use all)

  Usage
    eg: 

    from l1c_hsdi import l1c_hsdi
    l1c_hsdi ( 'l1b_hsdi.nc', 'hsdi.l1c' )

  """

  VERSID = '03MAY24'

  l1b = L1b_Hsdi ( l1bfil ) 
  nimg = l1b.nimg
  nchn = l1b.nchn
  
  if chnlst is None: chnlst = l1b.chn_lab
  if altmin is None: altmin = -999.0
  if altmax is None: altmax = 999.0

  with open ( l1cfil, 'w' ) as f:
    sp = ''  # dummy print variable for adding blank spaces
    f.write ( '! MORSE l1c file created by l1c_hsdi.py v' + VERSID + '\n' )

    f.write ( f'{3.3:10.1f}{sp:25}(=format id)\n' ) 

    f.write ( f'{2:10}{0.0:10.4f}{sp:15}(2=limb tra, resln)\n' )

    f.write ( f'{l1b.instrument:>10}{l1b.satellite:>10}{sp:15}' + \
               '(=Instrument,satellite)\n' )

    # Establish various time/date forms
    jday = l1b.julian_day[0]
    f.write ( f'{ymd(jday):>10}{jday:>10}{sp:15}' \
              '(=nominal date, day# since 1Jan2000)\n' )

    orb = f'{l1b.orbit:05}'    # fill out to 5 chars with leading zeros
    hmssta = hms ( l1b.milliseconds[0] )
    hmsend = hms ( l1b.milliseconds[nimg-1] )
    f.write ( f'{orb:>10}{hmssta:>10}{hmsend:>10}{sp:5}' + \
               '(=Orbit#, Event Start, End HHMMSS)\n') 

    f.write ( f'{1:10}{sp:25}(=No.scans)\n' )  # Single scan in file

    f.write ( f'{nimg:10}{"GEO":>10}{sp:15}' +
               '(=No.sweeps per scan, then nominal alts [km])\n' )

    # In L1B alts are in time order, but L1C always high to low
    idximg = np.arange ( nimg ) 
    if l1b.sunrise: idximg = idximg[::-1]

    alt = l1b.altitude
    if ( alt[idximg[0]] < alt[idximg[nimg-1]] ):
      print('F-l1c_hsdi: L1B altitude sequence does not match SUNRISE flag')
      exit()
       
    # Write list of tangent altitudes
    ifld = 0
    for idx in idximg:
      f.write ( f'{alt[idx]:8.3f}' )
      ifld += 1 
      if ifld % 10 == 0: f.write ('\n')
    if ifld % 10 != 0: f.write ('\n')    # ensure <CR> after last field

    ISCN = 1
    # Single scan in HSDI file
    f.write( f'{ISCN:10}{sp:5}(=Scan#)\n' )

    # Fixed values for these fields
    LST = 0.0
    SZA = 90.0
    CLDRAD = 0.0
    CLDIDX = 0.0

    img2 = 0
    for idx in idximg:
      f.write ( '!yyyymmdd hhmmss  millisec Scn Swp   Lat     Lon' + 
                    '   LST     SZA   CldRad CldIdx\n' )
      ymdalt = ymd ( l1b.julian_day[idx] ) 
      f.write ( f'{ymdalt:>9}' )
      msec = l1b.milliseconds[idx]
      hmsalt = hms(msec) 
      lat = l1b.latitude[idx] 
      lon = l1b.longitude[idx] 
      f.write ( f'{hmsalt:>7}' )
      f.write ( f'{msec:10}' )
      f.write ( f'{ISCN:4}' )
      f.write ( f'{idx+1:4}' )
      f.write ( f'{lat:7.2f}' )
      f.write ( f'{lon:8.2f}' )
      f.write ( f'{LST:7.2f}' )
      f.write ( f'{SZA:7.2f}' )
      f.write ( f'{CLDRAD:8.1f}' )
      f.write ( f'{CLDIDX:7.2f}' )
      f.write ( '\n' )

      # Construct list of L1B measurements to be listed in L1C file
      reclst = []
      img1 = img2 
      img2 += l1b.nuse[idx] 
      alt = l1b.altitude[idx]
      for ichn in range(nchn):
        chn_lab = l1b.chn_lab[ichn] 
        if chn_lab in chnlst:
          for idat in range(img1,img2):
            imos = l1b.idx_mos[idat]
            altrel = l1b.chn_alt[ichn] + l1b.mos_alt[imos]
            if alt + altrel >= altmin and alt + altrel <= altmax:
              tra = l1b.transmittance[idat,ichn]
              noi = l1b.noise[idat,ichn]
              mos_x = l1b.mos_x[imos]
              mos_y = l1b.mos_y[imos]
              rec = f'{chn_lab:<8}{altrel:10.3f}{tra:10.6f}{noi:10.6f}' + \
                    f'{mos_x:6}{mos_y:6}' + '\n'
              reclst.append(rec)
      alt = l1b.altitude[idx]
      rad = l1b.rad_curve[idx]
      nmeas = len(reclst)
      f.write ( '!NMeas AltNOM  AltImg  RadCrv\n' )
      f.write ( f'{nmeas:5}{alt:8.3f}{alt:8.3f}{rad:9.3f}\n' )
      if nmeas > 0: 
        f.write ( '! Chn_ID   Alt_Rel   Transm.     Noise Mos_X Mos_Y\n' )
        for rec in reclst: f.write ( rec )

l1c_hsdi ( 'l1b_hsdi.nc', 'test.l1c', altmin=5, altmax=60 )
