import numpy as np
from netCDF4 import Dataset
from ncvardef import ncvardef

class L2: 
  """ Data and Methods for HIROS/HSDI L2 structure

  Version
    20NOV24 AD Corrections to .latitude and .longitude
               Add self.ymd and self.hms
    30JUL24 AD Adapted from IDL module

  Usage
    from l2_class import L2

    l2 = L2 ( l2fil )  # load existing netCDF L2 file
    l2 = L2 ( )        # return empty L2 data object
    l2.write ( l1fil ) # write L2 data object to netCDF file

  Attributes
 
  Public Methods
    write    : Write L2 data structure to netCDF file

  Private Methods
    __init__ : Initialisation returns L1B structure
    _null    : Make empty L2 data structure
    _read    : Read L2 netCDF file and return as data structure
  """

  def __init__ ( self, l2fil=None ):
    """ Initialise 

    Parameters
      l2fil str : name of netCDF L2 file to be read (optional)

    Returns
      l2 object
    """    

    if l2fil:
      self._read ( l2fil )
    else:
      self._null ( ) 

  def _null ( self ):
    """  Make empty L2 data structure  """

    self.title    = None
    self.created  = None
    self.source   = None
    # Array dimensions
    self.nhgt = None
    # Scalars
    self.satellite    = None
    self.instrument   = None
    self.orbit        = None
    self.julian_day   = None
    self.milliseconds = None
    self.ymd          = None
    self.hms          = None
    self.latitude     = None
    self.longitude    = None

    self.quality      = None
    self.hgt = None
    self.labprf = None
    self.profiles = None

  def _read ( self, l2fil ):
    """ Read L2 structure from netCDF file
  
    Parameters
      l2fil str : Name of netCDF file
    """

    nc = Dataset ( l2fil )

  # Global attributes
    self.title    = nc.getncattr('Title')
    self.created  = nc.getncattr('Created')
    self.source   = nc.getncattr('Source')
    # Array dimensions
    self.nhgt = len ( nc.dimensions['NHgt'] )
    # Scalars
    self.satellite    = nc['Satellite'][0]
    self.instrument   = nc['Instrument'][0]
    self.orbit        = nc['Orbit'][0]
    self.julian_day   = nc['Julian_Day'][0]  
    self.milliseconds = nc['Milliseconds'][0]
    self.latitude     = nc['Latitude'][0]
    self.longitude    = nc['Longitude'][0]
    # Arrays dimensioned nmic
    self.mic_lab      = nc['Mic_Lab'][:]   # size nmic
    self.mic_npt      = nc['Mic_Npt'][:]
    self.mic_min      = nc['Mic_Min'][:]
    self.mic_max      = nc['Mic_Max'][:]
    self.mic_res      = nc['Mic_Res'][:]
    # Arrays dimensioned nalt
    self.rad_curve    = nc['Rad_Curve'][:]
    # Arrays dimensioned nmic,nalt
    self.quality      = nc['Quality'][:,:]
    # Arrays dimensioned nmic,nmax
    self.noise        = nc['Noise'][:,:]  # size nmic,nmax
    # Array dimensioned nmic,nalt,nmax
    self.transmittance = nc['Transmittance'][:,:,:]


  """ Create Create L2 netCDF file from MORSE .rtv file


  Arguments
    rtvfil  str  : Input .rtv filename
    ncfil   str  : Output netCDF filename

  """

  def write ( self, l2fil ):
    """ Write HIROS L2 structure to netCDF file

    Parameters
      l2fil str : netCDF filename

    Returns
      None
    """

    # Open output file
    # don't use NETCDF4_CLASSIC' so can use 'str'
    nc = Dataset ( l2fil, mode='w' ) 

    # Define Global attributes
    nc.Title   = self.title
    nc.Created = self.created
    nc.Source  = self.source

    # Define Array dimensions
    nc.createDimension('NHgt',self.nhgt)

    # Scalar Variables
    sat_ncv = nc.createVariable ( 'Satellite', str )
    sat_ncv[0] = self.satellite

    ins_ncv = nc.createVariable ( 'Instrument', str )
    ins_ncv[0] = self.instrument

    day_ncv = nc.createVariable ( 'julian_day', int )
    day_ncv[0] = self.julian_day

    ncvardef ( nc, 'orb', (), self.orbit ) 
    ncvardef ( nc, 'ymd', (), self.ymd ) 
    ncvardef ( nc, 'hms', (), self.hms ) 
    ncvardef ( nc, 'lat', (), self.latitude ) 
    ncvardef ( nc, 'lon', (), self.longitude ) 
    ncvardef ( nc, 'hgt', 'NHgt', self.hgt ) 

    for prf in self.profiles:
      if np.isscalar ( self.profiles[prf] ):
        nlvp = 1
      else:
        nlvp = len ( self.profiles[prf] ) 
      if nlvp > 1:
        ncvardef ( nc, prf, 'NHgt', self.profiles[prf] )
      else:
        ncvardef ( nc, prf, (), self.profiles[prf] )
