FUNCTION READ_AVK, AVKFIL, MPIX=MPIX, IDXSET=IDXSET
;
; VERSION
;   18NOV22 AD Original. Based on read_rtv.pro
;
; DESCRIPTION
;   IDL function to read MORSE .avk file
;
; ARGUMENTS
;   STRING  AVKFIL  I  Name of .avk file
;   LONG    MPIX    I  (Optional) No. of pixels to read (0=just header)
;   INTEGER IDXSET  I  (Optional) Index of set to be read (default=Final)
;
; RETURNS
;   Structure:

;
OPENR, LUN, AVKFIL, /GET_LUN

; Save first two comment records at start of file as HDR1 and HDR2
REC = '!'
IHDR = 0
HDR1 = ''
HDR2 = ''
WHILE NOT EOF ( LUN ) DO BEGIN
  READF, LUN, REC
  IF STRMID ( REC, 0, 1 ) EQ '!' THEN BEGIN
    IHDR = IHDR + 1
    IF IHDR EQ 1 THEN HDR1 = STRMID(REC,1)  ; remove initial '!' character
    IF IHDR EQ 2 THEN HDR2 = STRMID(REC,1)
  ENDIF ELSE BREAK
ENDWHILE
; At this point REC contains the first non-comment record of the file

; Read file format ID 
RFMT = 0.0
READS, REC, RFMT
IF RFMT GT 2.00 THEN $
  PRINT, 'W-READ_AVK: May not read new file format v' + $
         STRTRIM(REC,2) + ' correctly'

; Read viewing geometry ID 
IGEOM = 0
READF, LUN, IGEOM
IF IGEOM LE 1 OR IGEOM GE 4 THEN MESSAGE, $
  'F-READ_AVK: Undefined viewing geometry, IGEOM=', IGEOM
NADIR = IGEOM EQ 3  ; Nadir radiance
TRANS = IGEOM EQ 2  ; Limb transmittance
                    ; Else 1, for Limb radiance

; Read Instrument, Satellite 
READF, LUN, REC
S = STRSPLIT ( REC, /EXTRACT ) 
INSTRUMENT = S[0]
SATELLITE  = S[1]

; Read date as yyyymmdd and Julian Day since 1Jan2000
YMD = 0L
DAY = 0L
READF, LUN, YMD, DAY

; Read Orbit#, and start/end times as hhmmss
ORBIT  = 0L
HMSSTA = 0L
HMSEND = 0L
READF, LUN, ORBIT, HMSSTA, HMSEND

; Read No.pixels/profile locations and No. retrieval sets/pixel
NPIX = 0L
NSET = 0
READF, LUN, NPIX, NSET

; If MPIX keyword is set, only read as far as MPIX
IF KEYWORD_SET(MPIX) THEN NPIX = MIN ( [NPIX,MPIX] ) 

; Default is to read last/only set (final result) 
IF KEYWORD_SET(IDXSET) THEN BEGIN
  IF IDXSET LT 1 THEN MESSAGE, 'F-READ_AVK: Keyword IDXSET must be GE 1'
  IF IDXSET GT NSET THEN MESSAGE, 'F-READ_AVK: Keyword IDXSET=' + $
    STRING(IDXSET) + ' > No. sets in file =' + STRING(NSET) 
  MSET = IDXSET-1
ENDIF ELSE MSET = NSET-1

; Read No.retrieval levels, No.profiles (all types)
NLEV = 0
NPRF = 0
READF, LUN, NLEV, NPRF

READF, LUN, REC
S = STRSPLIT ( REC, /EXTRACT ) 
LABGRD = STRMID ( S[0], 1 ) 
LEVGRD = FLTARR(NLEV)
READF, LUN, LEVGRD

LABPRF = STRARR(NPRF)
NLVPRF = INTARR(NPRF)
SCAPRF = BYTARR(NPRF) 
IOFPRF = INTARR(NPRF)
ILVPRF = []
IOFF = 0
FOR IPRF = 0, NPRF-1 DO BEGIN
  IOFPRF[IPRF] = IOFF
  READF, LUN, REC
  S = STRSPLIT ( REC, /EXTRACT )
  LABPRF[IPRF] = S[0]
  READS, S[1], NLV
  NLVPRF[IPRF] = NLV
  IF NLV LT NLEV THEN BEGIN
    IF NLV EQ 0 THEN BEGIN
      NLVPRF[IPRF] = 1
      SCAPRF[IPRF] = 1
      ILVPRF = [ ILVPRF, 0 ]
    ENDIF ELSE BEGIN            ; Separate list of profile levels for retreival
      FLGLEV = INTARR(NLEV)
      READF, LUN, FLGLEV            ; list of NLEV 0 or 1 values
      IDXLEV = WHERE ( FLGLEV )     ; 1=retrieved at this level
      ILVPRF = [ ILVPRF, IDXLEV ]   
    ENDELSE
  ENDIF ELSE BEGIN 
    ILVPRF = [ ILVPRF, INDGEN(NLEV) ]
  ENDELSE
  IOFF = IOFF + NLVPRF[IPRF]
ENDFOR
READF, LUN, REC   ; record marking end of header 

HDR = { HDR1:HDR1, $
        HDR2:HDR2, $
       IGEOM:IGEOM, $
  INSTRUMENT:INSTRUMENT, $
   SATELLITE:SATELLITE, $
         YMD:YMD, $
         DAY:DAY, $
       ORBIT:ORBIT, $
      HMSSTA:HMSSTA, $
      HMSEND:HMSEND, $
        NPIX:NPIX, $
        NLEV:NLEV, $
        NPRF:NPRF, $
      IDXSET:MSET, $
      LABGRD:LABGRD, $
      LEVGRD:LEVGRD, $
      LABPRF:LABPRF, $
      NLVPRF:NLVPRF, $
      IOFPRF:IOFPRF, $
      SCAPRF:SCAPRF, $
      ILVPRF:ILVPRF }

IF NPIX EQ 0 THEN BEGIN
  FREE_LUN, LUN
  RETURN, HDR  
ENDIF 

NPTTOT = TOTAL(NLVPRF)
AVK = FLTARR(NPTTOT,NPTTOT)
IF NADIR THEN $
  PIX = REPLICATE ( { YMD:0L, HMS:0L, MSC:0L, STP:0, FOV:0, $
                      LAT:0.0, LON:0.0, ZEN:0.0, SZA:0.0, CLD:0.0, LND:0.0, $
                      AVK:AVK }, NPIX )  $
ELSE $
  PIX = REPLICATE ( { YMD:0L, HMS:0L, MSC:0L, LAT:0.0, LON:0.0, $
                      LST:0.0, SZA:0.0, AVK:AVK }, NPIX)

IYMD = 0L
IHMS = 0L
IMSC = 0L
ISTP = 0
IFOV = 0
RLAT = 0.0
RLON = 0.0
RZEN = 0.0
RSZA = 0.0
RCLD = 0.0
RLND = 0.0
RLST = 0.0

JPIX = 0
JSET = 0
FOR IPIX = 0L, NPIX-1 DO BEGIN
  READF, LUN, JPIX
  READF, LUN, REC
  IF NADIR THEN READF, LUN, $
    IYMD, IHMS, IMSC, ISTP, IFOV, RLAT, RLON, RZEN, RSZA, RCLD, RLND $
  ELSE READF, LUN, $
    IYMD, IHMS, IMSC, RLAT, RLON, RSZA, RLST
  PIX[IPIX].YMD = IYMD
  PIX[IPIX].HMS = IHMS
  PIX[IPIX].MSC = IMSC
  PIX[IPIX].LAT = RLAT 
  PIX[IPIX].LON = RLON 
  PIX[IPIX].SZA = RSZA
  IF NADIR THEN BEGIN
    PIX[IPIX].ZEN = RZEN 
    PIX[IPIX].STP = ISTP
    PIX[IPIX].FOV = IFOV
    PIX[IPIX].CLD = RCLD
    PIX[IPIX].LND = RLND
  ENDIF ELSE PIX[IPIX].LST = RLST

  DAT = [ ]
  FOR ISET = 0, NSET-1 DO BEGIN
    READF, LUN, REC    ; skip set header
    READF, LUN, AVK
    IF ISET EQ MSET THEN PIX[IPIX].AVK = AVK
    READF, LUN, REC    ; '*END' record
  ENDFOR      
ENDFOR

FREE_LUN, LUN

RETURN, { HDR:HDR, PIX:PIX } 

END

avkfil = 'hiros_ch4.avk'
AVK = READ_AVK(avkfil)
end
