FUNCTION READ_L1B_HIROS,  NCFILE
;
; VERSION
;   14JUN24 AD Change NOISE from (NALT,NMIC) to (NMAX,NMIC)
;   26JUN23 AD Replace MIC_ALT(:) with ALT_OFFSET(:,:)
;   21APR23 AD Add MIC_ALT, ALT_TREND and ALT_QUAD (may or may not be in NCFILE)
;   18OCT22 AD Change MIC_MIN,MAX to Double precision (comment change only)
;   31MAY22 AD Read in Quality flags
;   12JAN22 AD Change RAD_CURV to RAD_CURVE
;   02DEC21 AD New Noise structure
;   01AUG21 AD Comment changes
;   22JUL21 AD Rename Band to Microwindow. Rename local variables for consistency
;   19JUL21 AD Add Band_ID to structure
;   14JUL21 AD Original
;
; DESCRIPTION
;   IDL function to read HIROS L1B netCDF file
;   Returns file contents as Structure (see bottom of code for structure)
;
; ARGUMENTS
;   NCFILE  String  I  Name of netCDF file to be read
;
; RETURNS
;   Structure containing:
;   .TITLE          String       Title from file
;   .CREATED        String       Creation date of file
;   .SOURCE         String       Source of data
;   .NMIC           Int          Number of different spectral bands
;   .NALT           Int          Number of different tangent alts for spectra
;   .NMAX           Long         Maximum No spectral points in any band
;   .SATELLITE      String       Satellite ID
;   .INSTRUMENT     String       Instrument ID
;   .ORBIT          Long         Orbit number
;   .SUNRISE        Byte         1=sunrise, 0=sunset
;   .MIC_LAB        String(NMIC) Microwindow identifier: 'A','B' or 'C'
;   .MIC_NPT        Long(NMIC)   No. spectral points in each Mic
;   .MIC_MIN        Double(NMIC) Min Wno [cm-1] for each Mic
;   .MIC_MAX        Double(NMIC) Max Wno [cm-1] for each Mic
;   .MIC_RES        Float(NMIC)  Wno resolution [cm-1] for each Mic
;   .JULIAN_DAY     Long(NALT)   Julian Day#
;   .MILLISECONDS   Long(NALT)   Milliseconds
;   .ALTITUDE       Float(NALT)  Geometric tangent altitude [km]
;   .LATITUDE       Float(NALT)  Tangent point latitude [deg N]
;   .LONGITUDE      Float(NALT)  Tangent point longitude [deg E]
;   .RAD_CURVE      Float(NALT)  Earth radius of curvature [km] in LOS plane
;   .NOISE          Float(NALT,NMIC)  Noise estimate for each Alt,Mic
;   .ALT_OFFSET     Float(NALT,NMIC)  Microwindow altitude offset [km]
;   .ALT_TREND      Float(NALT,NMIC)  Linear drift in tan.pt [km]
;   .ALT_QUAD       Float(NALT,NMIC)  Quadratic drift in tan.pt [km]
;   .QUALITY        Long(NALT,NMIC)   Quality flags (0=OK)
;   .TRANSMITTANCE  Float(NMAX,NALT,NMIC) Transmittance spectra
;
; USAGE
;   eg l1b = READ_L1B_HIROS ( 'l1b_hiros.nc' )
;      print, l1b.title

; Open netCDF file as NCID
NCID = NCDF_OPEN ( NCFILE )

; Get Global Attributes
NCDF_ATTGET, NCID, 'Title', TITLE, /GLOBAL
TITLE=STRING(TITLE)
NCDF_ATTGET, NCID, 'Created', CREATED, /GLOBAL
CREATED = STRING(CREATED)
NCDF_ATTGET, NCID, 'Source', SOURCE, /GLOBAL
SOURCE = STRING(SOURCE)

A = NCDF_INQUIRE ( NCID )  ; get list of array dimensions

FOR IDIM = 0, A.NDIMS-1 DO BEGIN
  NCDF_DIMINQ, NCID, IDIM, NAM, ISIZ
  CASE NAM OF
    'NMic': NMIC = ISIZ  ; No.microwindows
    'NAlt': NALT = ISIZ  ; No.tangent altitudes
    'NMax': NMAX = ISIZ  ; Max No.spectral points
  ENDCASE
ENDFOR
; Note that NMAX is the *maximum* number of spectral points in any microwindow

; Read scalar variables
NCDF_VARGET, NCID, 'Satellite', SATELLITE
SATELLITE = STRING(SATELLITE)
NCDF_VARGET, NCID, 'Instrument', INSTRUMENT
INSTRUMENT = STRING(INSTRUMENT)
NCDF_VARGET, NCID, 'Orbit', ORBIT
NCDF_VARGET, NCID, 'Sunrise', SUNRISE

; Read NMIC-size arrays
NCDF_VARGET, NCID, 'Mic_Lab', MIC_LAB  ; Microwindow Label ['A','B','C']
NCDF_VARGET, NCID, 'Mic_Min', MIC_MIN  ; Lower wavenumber [cm-1]
NCDF_VARGET, NCID, 'Mic_Max', MIC_MAX  ; Upper wavenumber [cm-1]
NCDF_VARGET, NCID, 'Mic_Res', MIC_RES  ; Spectral sampling [cm-]1
NCDF_VARGET, NCID, 'Mic_Npt', MIC_NPT  ; No.points in microwindow

; Read NALT-size arrays
NCDF_VARGET, NCID, 'Julian_Day', DAYLST   ; List of Julian Days
NCDF_VARGET, NCID, 'Milliseconds', MSCLST ; List of milliseconds
NCDF_VARGET, NCID, 'Altitude', ALTLST     ; List of tangent altitudes [km]
NCDF_VARGET, NCID, 'Latitude', LATLST     ; List of tangent latitudes [deg N]
NCDF_VARGET, NCID, 'Longitude', LONLST    ; List of tangent longitudes [deg E]
NCDF_VARGET, NCID, 'Rad_Curve', CRVLST    ; List of radii [km]

; Alt_Offset was introduced 01JUN23 
IF NCDF_VARID ( NCID, 'Alt_Offset' ) NE -1 $   ; If present in L1B data
  THEN NCDF_VARGET, NCID, 'Alt_Offset', ALT_OFFSET $ ; Then read Alt.offset [km]
  ELSE ALT_OFFSET = FLTARR(NALT,NMIC)                ; Else set to 0

; Since these are new additions (11APR23) to the L1B format they may not be
; present in the existing L1B data, in which case calculate values
IF NCDF_VARID ( NCID, 'Alt_Trend' ) EQ -1 THEN BEGIN
  ALT_TREND = FLTARR(NALT,NMIC)
  ALT_QUAD = FLTARR(NALT,NMIC) 
  FOR IALT = 0, NALT-1 DO BEGIN
    IF IALT EQ 0 THEN Z1 =-0.5 * ( ALTLST[IALT+1] - ALTLST[IALT] ) $
                 ELSE Z1 = 0.5 * ( ALTLST[IALT-1] - ALTLST[IALT] ) 
    IF IALT EQ NALT-1 THEN Z3 =-0.5 * ( ALTLST[IALT-1] - ALTLST[IALT] ) $
                      ELSE Z3 = 0.5 * ( ALTLST[IALT+1] - ALTLST[IALT] ) 
    ALT_TREND[IALT,*] = Z3 - Z1
    ALT_QUAD[IALT,*] = 2 * ( Z3 + Z1 )
  ENDFOR
ENDIF ELSE NCDF_VARGET, NCID, 'Alt_Trend', ALT_TREND ; Altitude trends [km]

; Assume either both Alt_Trend and Alt_Quad are present, or neither.
IF NCDF_VARID ( NCID, 'Alt_Quad' ) NE -1 THEN $
  NCDF_VARGET, NCID, 'Alt_Quad', ALT_QUAD ; List of Alt. Quad. trends [km]

; Read Quality data, dimensions (nalt,nmic)
NCDF_VARGET, NCID, 'Quality', QUALITY ; Microwindow quality flags

; Read Noise data, dimensions (nmax,nmic) ( was (nalt,nmic) )
NCDF_VARGET, NCID, 'Noise', NOI  ; Microwindow noise value

; Allow for old-format L1B files still containing noi as (nalt,nmic)
asiz = size(noi)
if asiz[1] eq nalt and asiz[1] ne nmax then begin
  print,'W-READ_L1B_HIROS: Resizing NOISE from (NALT,NMIC) to (NMAX,NMIC)'
  noiold = noi
  noi = fltarr(nmax,nmic)
  for imic = 0, nmic-1 do begin
    noi[*,imic] = noiold[0,imic]
  endfor
endif

; Read Transmittance spectra, dimensions (nspc,nalt,nmic)
NCDF_VARGET, NCID, 'Transmittance', TRA   

NCDF_CLOSE, NCID

RETURN, { TITLE:TITLE,   $   ; Title from file
        CREATED:CREATED, $   ; Creation date of file
         SOURCE:SOURCE,  $   ; Source of data
           NMIC:NMIC,    $   ; Number of different spectral bands
           NALT:NALT,    $   ; Number of different tangent altitudes for spectra
           NMAX:NMAX,    $   ; Maximum No spectral points in any band
      SATELLITE:SATELLITE, $ ; Satellite ID
     INSTRUMENT:INSTRUMENT,$ ; Instrument ID
          ORBIT:ORBIT,   $   ; Orbit number
        SUNRISE:SUNRISE, $   ; 1=sunrise, 0=sunset
        MIC_LAB:MIC_LAB, $   ; (nmic) Microwindow identifier: 'A','B' or 'C'
        MIC_NPT:MIC_NPT, $   ; (nmic) No. spectral points in each Mic
        MIC_MIN:MIC_MIN, $   ; (nmic) Min Wno [cm-1] for each Mic
        MIC_MAX:MIC_MAX, $   ; (nmic) Max Wno [cm-1] for each Mic
        MIC_RES:MIC_RES, $   ; (nmic) Wno resolution [cm-1] for each Mic
     JULIAN_DAY:DAYLST,  $   ; (nalt) List of Julian Day#
   MILLISECONDS:MSCLST,  $   ; (nalt) List of milliseconds
       LATITUDE:LATLST,  $   ; (nalt) List of tangent latitudes [deg N]
      LONGITUDE:LONLST,  $   ; (nalt) List of tangent longitudes [deg E]
      RAD_CURVE:CRVLST,  $   ; (nalt) List of radii of curvature [km]
       ALTITUDE:ALTLST,  $   ; (nalt) List of tangent altitudes [km]
     ALT_OFFSET:ALT_OFFSET,$ ; (nalt,nmic) List of altitude offsets [km]
      ALT_TREND:ALT_TREND, $ ; (nalt,nmic) List of altitude trends [km]
       ALT_QUAD:ALT_QUAD,  $ ; (nalt,nmic) List of altitude quadratic trends [km]
        QUALITY:QUALITY, $   ; (nalt,nmic) Quality flags
          NOISE:NOI,     $   ; (nmax,nmic) Noise spectrum for each mic
  TRANSMITTANCE:TRA        } ; (nmax,nalt,nmic) transmittance spectra

END
