FUNCTION READ_L1B_HSDI,  NCFILE
;
; VERSION
;   22JAN22 AD Add Quality and Noise data. Change RAD_CURV to RAD_CURVE
;   21JUL21 AD Original
;
; DESCRIPTION
;   IDL function to read HSDI L1B netCDF file
;   Returns file contents as Structure (see bottom of code for structure)
;
; ARGUMENTS
;   NCFILE  String  I  Name of netCDF file to be read
;
; RETURNS
;   Structure:
;   .TITLE          String            Title from file
;   .CREATED        String            Creation date of file
;   .SOURCE         String            Source of data
;   .NIMG           Long              No. Images
;   .NCHN           Long              No. Channels
;   .NMOS           Long              No. Mosaics
;   .NDAT           Long              No. Data points
;   .SATELLITE      String            Satellite ID
;   .INSTRUMENT     String            Instrument ID
;   .ORBIT          Long              Orbit #
;   .SUNRISE        Byte              1=sunrise  0=sunset
;   .CHN_LAB        String(NCHN)      Channel identifier
;   .CHN_X          Int(NCHN)         Channel column within mosaic
;   .CHN_Y          Int(NCHN)         Channel row within mosaic
;   .CHN_ALT        Float(NCHN)       Channel relative tangent altitude [km]
;   .MOS_X          Float(NMOS)       x-coordinate of mosaic within image
;   .MOS_Y          Float(NMOS)       y-coordinate of mosaic within image
;   .MOS_ALT        Float(NMOS)       Tangent altitude displacement [km] of mosaic
;   .JULIAN_DAY     Long(NIMG)        Day since 1st Jan 2000
;   .MILLISECONDS   Long(NIMG)        Milliseconds since midnight
;   .ALTITUDE       Float(NIMG)       Geometric tangent point altitude [km]
;   .LATITUDE       Float(NIMG)       Tangent point longitude [degN]
;   .LONGITUDE      Float(NIMG)       Tangent point longitude [degE]
;   .RAD_CURVE      Float(NIMG)       Earth radius of curvature [km] in LOS plane      
;   .NUSE           Int(NIMG)         No. mosaics used for each image
;   .IDX_MOS        Int(NDAT)         Indices of mosaics used for images
;   .QUALITY        Long(NCHN,NDAT)   Quality flags (0=OK)
;   .NOISE          Float(NCHN,NDAT)  Noise data
;   .TRANSMITTANCE  Float(NCHN,NDAT)  Transmittance data

;
; USAGE
;   eg HDATA = READ_L1B_HSDI ( 'l1b_hsdi.nc' )

NCID = NCDF_OPEN ( NCFILE )

; Get Global Attributes
NCDF_ATTGET, NCID, 'Title', TITLE, /GLOBAL
TITLE=STRING(TITLE)
NCDF_ATTGET, NCID, 'Created', CREATED, /GLOBAL
CREATED = STRING(CREATED)
NCDF_ATTGET, NCID, 'Source', SOURCE, /GLOBAL
SOURCE = STRING(SOURCE)

A = NCDF_INQUIRE ( NCID )  ; get list of array dimensions

FOR IDIM = 0, A.NDIMS-1 DO BEGIN
  NCDF_DIMINQ, NCID, IDIM, NAME, ISIZ
  CASE NAME OF
    'NImg': NIMG = ISIZ  ; No. Images
    'NChn': NCHN = ISIZ  ; No. spectral channels
    'NMos': NMOS = ISIZ  ; No. different mosaics defined
    'NDat': NDAT = ISIZ  ; No. mosaic datapoints in file
  ENDCASE
ENDFOR

; Read scalar variables
NCDF_VARGET, NCID, 'Satellite', SATELLITE
SATELLITE = STRING(SATELLITE)
NCDF_VARGET, NCID, 'Instrument', INSTRUMENT
INSTRUMENT = STRING(INSTRUMENT)
NCDF_VARGET, NCID, 'Orbit', ORBIT
NCDF_VARGET, NCID, 'Sunrise', SUNRISE

; Read NCHN-size arrays
NCDF_VARGET, NCID, 'Chn_Lab', CHN_LAB ; Channel labels [eg '01', '02' etc]
NCDF_VARGET, NCID, 'Chn_X',   CHN_X   ; Channel col# within mosaic (1=left)
NCDF_VARGET, NCID, 'Chn_Y',   CHN_Y   ; Channel row# within mosaic (1=top)
NCDF_VARGET, NCID, 'Chn_Alt', CHN_ALT ; Channel relative altitude [km]

; Read NMOS-size arrays
NCDF_VARGET, NCID, 'Mos_X',   MOS_X   ; Mosaic col# within image (1=left)
NCDF_VARGET, NCID, 'Mos_Y',   MOS_Y   ; Mosaic row# within image (1=top)
NCDF_VARGET, NCID, 'Mos_Alt', MOS_ALT ; Mosaic relative altitude [km]

; Read NIMG-size arrays
NCDF_VARGET, NCID, 'Julian_Day',   JULIAN_DAY   ; List of Julian Days
NCDF_VARGET, NCID, 'Milliseconds', MILLISECONDS ; List of milliseconds
NCDF_VARGET, NCID, 'Altitude',     ALTITUDE     ; List of tangent altitudes [km]
NCDF_VARGET, NCID, 'Latitude',     LATITUDE     ; List of tangent latitudes [deg N]
NCDF_VARGET, NCID, 'Longitude',    LONGITUDE    ; List of tangent longitudes [deg E]
NCDF_VARGET, NCID, 'Rad_Curve',    RAD_CURVE    ; List of radii [km]
NCDF_VARGET, NCID, 'NUse',         NUSE         ; No.mosaics used in each image

; Read NDAT-size
NCDF_VARGET, NCID, 'Idx_Mos', IDX_MOS  ; Indices of mosaics for each data point

; Read Quality flags (nchn,ndat)
NCDF_VARGET, NCID, 'Quality', QUALITY

; Read Noise values (nchn,ndat)
NCDF_VARGET, NCID, 'Noise', NOISE

; Read Transmittance values, dimensions (nchn,ndat)
NCDF_VARGET, NCID, 'Transmittance', TRANSMITTANCE   

NCDF_CLOSE, NCID

RETURN, {                      $
          TITLE:TITLE,         $ ; Title from file
        CREATED:CREATED,       $ ; Creation date of file
         SOURCE:SOURCE,        $ ; Source of data
           NIMG:NIMG,          $ ; No. Images
           NCHN:NCHN,          $ ; No. Channels
           NMOS:NMOS,          $ ; No. Mosaics
           NDAT:NDAT,          $ ; No. Data points
      SATELLITE:SATELLITE,     $ ; Satellite ID
     INSTRUMENT:INSTRUMENT,    $ ; Instrument ID
          ORBIT:ORBIT,         $ ; Orbit #
        SUNRISE:SUNRISE,       $ ; 1=sunrise, 0=sunset
        CHN_LAB:CHN_LAB,       $ ; (nchn) Channel labels (eg '01', '02 ...)
          CHN_X:CHN_X,         $ ; (nchn) Channel col# within mosaic
          CHN_Y:CHN_Y,         $ ; (nchn) Channel row# within mosaic
        CHN_ALT:CHN_ALT,       $ ; (nchn) Channel relative altitude
          MOS_X:MOS_X,         $ ; (nmos) Mosaic col# within image
          MOS_Y:MOS_Y,         $ ; (nmos) Mosaic row# within image
        MOS_ALT:MOS_ALT,       $ ; (nmos) Mosaic relative altitude
     JULIAN_DAY:JULIAN_DAY,    $ ; (nimg) List of Julian Day#
   MILLISECONDS:MILLISECONDS,  $ ; (nimg) List of milliseconds
       ALTITUDE:ALTITUDE,      $ ; (nimg) List of tangent altitudes [km]
       LATITUDE:LATITUDE,      $ ; (nimg) List of tangent latitudes [deg N]
      LONGITUDE:LONGITUDE,     $ ; (nimg) List of tangent longitudes [deg E]
      RAD_CURVE:RAD_CURVE,     $ ; (nimg) List of radii of curvature [km]
           NUSE:NUSE,          $ ; (nimg) No. mosaics used for each image
        IDX_MOS:IDX_MOS,       $ ; (ndat) indices of mosaics used
        QUALITY:QUALITY,       $ ; (nchn,ndat) quality flags (0=OK)
          NOISE:NOISE,         $ ; (nchn,ndat) noise data
  TRANSMITTANCE:TRANSMITTANCE  } ; (nchn,ndat) transmittance data
END

l1b = read_l1b_hsdi('l1b_hsdi.nc')

end
