FUNCTION READ_L1C, L1CFIL
;
; VERSION
;   27JAN23 AD Original.
;
; DESCRIPTION
;   IDL function to read MORSE L1C file into data structure
;   Since the number of microwindows within any sweep is variable, and
;   microwindows themselves are of different sizes, the microwindow headers
;   and microwindow spectral data are stored in separate arrays.
;
;   From SWPHDR for any sweep use IPTMIC to get index in MICDAT of 1st MW header
;   within the sweep (subsequent MW headers are stored sequentially).
;
;   From MICDAT for any microwindow/sweep, use IPTSPC to get offset of 
;   associated spectral data in SPCDAT.
;
;
; ARGUMENTS
;   L1CFIL  String  I  Name of .l1c file to be read
;
; RETURNS
;   Structure, see end of file for details
;
; USAGE
;   l1c = read_l1c ( 'file.l1c' )
;

OPENR, LUN, L1CFIL, /GET_LUN

; Skip any initial comment records, starting with '!' character
REC = '!'
WHILE STRMID ( REC, 0, 1 ) EQ '!' DO READF, LUN, REC

READS, REC, FORMAT_ID
IVIEW = 0
READF, LUN, IVIEW, RESLN

READF, LUN, REC
INSTRUMENT = STRTRIM ( STRMID(REC,0,10), 2 ) 
SATELLITE  = STRTRIM ( STRMID(REC,10,10), 2 )

YMD = 0L & JULIAN_DAY = 0L
READF, LUN, YMD, JULIAN_DAY

ORBIT = 0L & TIME_START = 0L & TIME_END   = 0L
READF, LUN, ORBIT, TIME_START, TIME_END

NSCAN = 0
READF, LUN, NSCAN

NSWEEP = 0
READF, LUN, NSWEEP
GRID = FLTARR(NSWEEP)
READF, LUN, GRID

L1CHDR = {                      $
        FORMAT_ID : FORMAT_ID,  $  ; File format identifier
            IVIEW : IVIEW,      $  ; Viewing geometry (2=limb tra)
            RESLN : RESLN,      $  ; Spectral resolution (0=filter channels
              YMD : YMD,        $  ; Year, month, day 
       JULIAN_DAY : JULIAN_DAY, $  ; Julian Day# (0=1st Jan 2000)
            ORBIT : ORBIT,      $  ; Orbit#
       TIME_START : TIME_START, $  ; Start Time of observations [hhmmss]
         TIME_END : TIME_END,   $  ; End Time of observations [hhmmss]
            NSCAN : NSCAN,      $  ; No. scans (meas.vert.profiles) in file 
           NSWEEP : NSWEEP,     $  ; No. sweeps (meas.steps) per scan 
             GRID : GRID }         ; Nominal grid levels [km] of measurements

IF L1CHDR.NSCAN NE 1 THEN MESSAGE, $
  'F-READ_L1C: this version for single scan L1C files only'
READF, LUN, REC     ; Scan Header record, just contains scan#1 starting with 1

; Define Sweep Header variables which are not Float
YMD = 0L & HMS = 0L & MSC = 0L
IDXSCN = 0 & IDXSWP = 0 & NMIC = 0
RADCRV = 0.0D0

; Define Microwindow Header variables which are not float
NWNO = 0
WNO1 = 0.0D0 & WNO2 = 0.0D0

SWPHDR = []  ; Array of structures containing header data for each sweep
MICDAT = []  ; Array of microwindow headers for each sweep/microwindow
SPCDAT = []  ; Concatenated array of MW spectra for entire scan

IPTMIC = 0   ; Offset for MW in MICDAT
IPTSPC = 0L  ; Offset for MW spectrum in SPCDAT

FOR ISWP = 0, NSWEEP-1 DO BEGIN
  READF, LUN, REC   ; header text
  READF, LUN, YMD, HMS, MSC, IDXSCN, IDXSWP, LAT, LON, LST, SZA, CLDRAD, CLDIDX
  READF, LUN, REC   ; header text
  READF, LUN, NMIC, ALTNOM, ALTSWP, RADCRV
  SWPHDR = [ SWPHDR, { YMD:YMD, HMS:HMS, MSC:MSC, IDXSCN:IDXSCN, IDXSWP:IDXSWP,$
                       LAT:LAT, LON:LON, LST:LST, SZA:SZA, $
                       CLDRAD:CLDRAD, CLDIDX:CLDIDX, $
                       NMIC:NMIC, IPTMIC:IPTMIC, ALTNOM:ALTNOM, ALTSWP:ALTSWP,$
                       RADCRV:RADCRV } ]
  IPTMIC = IPTMIC + NMIC
  FOR IMIC = 0, NMIC-1 DO BEGIN
    READF, LUN, REC
    MICLAB = STRTRIM ( STRMID ( REC, 0, 7 ), 2 ) 
    READS, STRMID ( REC, 8 ), NWNO, WNO1, WNO2, NESR
    MICDAT = [ MICDAT, { MICLAB:MICLAB, NWNO:NWNO, IPTSPC:IPTSPC, $
                         WNO1:WNO1, WNO2:WNO2, NESR:NESR, ISWP:ISWP } ]
    SPC = FLTARR(NWNO)
    READF, LUN, SPC
    SPCDAT = [ SPCDAT, SPC ]
    IPTSPC = IPTSPC + NWNO
  ENDFOR
ENDFOR

RETURN,  { L1CHDR:L1CHDR, SWPHDR:SWPHDR, MICDAT:MICDAT, SPCDAT:SPCDAT }

END

